; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   IDLitsrvReadFile
;
; PURPOSE:
;   This file extends IDL's IDLitsrvReadFile class to include
;   additional methods which enable searching of registered file
;   readers.
;
; CATEGORY:
;   IDL iTool
;
; SUPERCLASSES:
;
; SUBCLASSES:
;
; CREATION:
;
; METHODS:
;   _FindReadersByName
;   GetReadersByName
;   GetFilterListByReaderName
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; July 2004
;-
;###############################################################################


;===============================================================================
; IDLitsrvReadFile::_FindReadersByName
; 
; PURPOSE:
;   Search reader(s) (specified by an array of object descriptors) for
;   those that match the specified name(s). This is a private method
;   based on IDLitsrvWriteFile::_FindReadersByType
;
; PARAMETERS:
;   oDesc [in] - An array of object descriptors for the readers.
;
;   name [in] - The name(s) of the desired reader(s).
;
; KEYWORDS:
;   count [out] - The number of readers found
;
; RETURN VALUE:
;    If successful, an array of reader object descriptors
;    If unsuccessful, a null object.
;
function IDLitsrvReadFile::_FindReadersByName, name, oDesc, count=count
compile_opt idl2

count=0
;; just loop and do a type match check
for i=0, n_elements(oDesc)-1 do begin
    oDesc[i]->GetProperty, name=objName
    void = where(strcmp(name, objName, /FOLD_CASE) gt 0, nMatch)
    if (nMatch gt 0 or objName[0] eq '') then begin ;; add to list
        ret = (count eq 0)? oDesc[i] : [ret, oDesc[i]]
        count++
    endif
endfor

return, count gt 0 ? ret : obj_new()

end


;===============================================================================
; IDLitsrvReadFile::GetReadersByName
; 
; PURPOSE:
;   Get reader(s) (that are registered with the current or system
;   tool) that have the specified name.
;
; PARAMETERS:
;   name [in] - The name(s) of the reader(s) to be located. The name
;           refers to the IDLitComponent name that was given
;           to the reader. eg the IDLitReadAscii reader has the name
;           'ASCII text' which is specified in  IDLitReadAscii::Init.
;
; KEYWORDS:
;   count [out] - The number of readers found
;
; RETURN VALUE:
;    If successful, an array of object identifiers (strings) of the readers.
;    If unsuccessful, an empty string.
;
function IDLitsrvReadFile::GetReadersByName, name, count=count
compile_opt idl2

oTool = self->GetTool()

oDesc = oTool->GetFileReader(count=count,/all)
if (count eq 0) then return, ''

oDesc = self->_FindReadersByName(name, oDesc, count=count)
if (count eq 0) then return, '';

idRet = strarr(count)
for i=0, count-1 do idRet[i] = oDesc[i]->GetFullIdentifier()

return, idRet

end


;===============================================================================
; IDLitsrvReadFile::GetFilterListByReaderName
; 
; PURPOSE:
;   Find the list of filters (file extensions) that have been
;   registered for the specified reader name(s).
;
; PARAMETERS:
;   name [in] - The name(s) of the desired reader(s).
;
; KEYWORDS:
;   count [out] - The number of readers found
;
; RETURN VALUE:
;    If successful, an array of reader name/filter(s) pairs
;    If unsuccessful, an empty string.
;
function IDLitsrvReadFile::GetFilterListByReaderName, name, COUNT=COUNT

    compile_opt idl2, hidden

    oTool = self->GetTool()

    oDesc = oTool->GetFileReader(count=count, /all)
    oDesc = self->_FindReadersByName(name, oDesc, count=count)

    if (count gt 0) then begin
        self->BuildExtensions, oDesc, sExten, sFilter, sID
    endif else return, ''

    ;; If 'All files' was included then add the *.* filter spec.
    void = where(strcmp(name, 'All files', /FOLD_CASE) gt 0, nMatch)
    if (nMatch) then $
      sFilter = [sFilter, [['*.*'],['All files']]]

    sFilter[*,1] += ' (' + sFilter[*,0] + ')'

    count = N_ELEMENTS(sFilter)/2

    return, sFilter
end
