; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   ASCIISPEDataset
;
; PURPOSE:
;   Data Class for representing SPE formated datasets
;
; CATEGORY:
;
; SUPERCLASSES:
;   GenericDataset
;
; SUBCLASSES:
;
; CREATION:
;   See ASCIISPEDataset::Init
;
; METHODS:
;   GetProperty
;   SetProperty
;   ToDavePtr
;   ToASCIISPE
;   ToASCIIGrp
;   ToASCIIColumn
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; December 2008
;-
;###############################################################################


;===============================================================================



;===============================================================================
; ASCIISPEDataset::toASCIIGrp
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII Grp data format variables.
;
; PARAMETERS
;    xValue - the first independent variable
;    
;    yValue - the second independent variable
;    
;    dataValue - the dependent data
;    
;    errorValue - the uncertainty in the dependent data
;    
;    xTitle - the first independent axis label/title
;    
;    yTitle - the second indenpendent axis label/title
;    
;    dataTitle - the dependent label/title
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
;function ASCIISPEDataset::toASCIIGrp, xValue,yValue,dataValue,errorValue,xTitle,yTitle,dataTitle
;compile_opt idl2
;
;; need at least the first 3 parameters to proceed
;if (n_params() lt 3) then return, 0
;
;Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
;                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
;                 ,axis1Distribution=xType, axis2Distribution=yType
;
;; all relevant data must be present
;xExist = n_elements(xData) gt 0
;yExist = n_elements(yData) gt 0
;qtyExist = n_elements(qty) gt 0
;errPresent = n_elements(err) gt 0
;if (~xExist || ~yExist || ~qtyExist) then return, 0
;
;; must be a 2D dataset
;ndim = size(qty,/n_dimensions)
;if (ndim ne 2) then return, 0
;
;nx = n_elements(xData)
;ny = n_elements(yData)
;nqty1 = (size(qty))[1]
;nqty2 = (size(qty))[2]
;if (nx ne nqty1) then begin
;   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (nx eq nqty1+1)) then begin
;      ;; convert to point mode. ASCII grp format only stores point
;      ;; mode data
;     index = lindgen(nqty1)
;     xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries
;     nx = n_elements(xData)
;   endif else begin
;      ;msg = 'Size of dependent and first independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endelse
;endif
;if (ny ne nqty2) then begin
;   isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (ny eq nqty2+1)) then begin
;      ;; convert to point mode
;     index = lindgen(nqty2)
;     yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries
;     ny = n_elements(yData)
;   endif else begin
;      ;msg = 'Size of dependent and second independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endelse
;endif
;
;; Generate errors if not present
;if (~errPresent) then begin
;   index = where(qty le 0, cnt)
;   err = qty
;   if (cnt gt 0) then err[index] = 1.0
;   err = temporary(sqrt(err))
;endif
;
;xValue = temporary(xData)
;yValue = temporary(yData)
;dataValue = temporary(qty)
;errorValue = temporary(err)
;xTitle = xLabel
;yTitle = yLabel
;dataTitle = qtyLabel
;
;return, 1
;
;end



;===============================================================================
; ASCIISPEDataset::toASCIISPE
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII SPE data format variables.
;
; PARAMETERS
;    xValue - the first independent variable
;    
;    yValue - the second independent variable
;    
;    dataValue - the dependent data
;    
;    errorValue - the uncertainty in the dependent data
;    
;    xTitle - the first independent axis label/title
;    
;    yTitle - the second indenpendent axis label/title
;    
;    dataTitle - the dependent label/title
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
;function ASCIISPEDataset::toASCIISPE, xValue,yValue,dataValue,errorValue,xTitle,yTitle,dataTitle
;compile_opt idl2
;
;; need at least the first 3 parameters to proceed
;if (n_params() lt 3) then return, 0
;
;retVal = 1
;
;Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
;                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
;                 ,axis1Distribution=xType, axis2Distribution=yType
;
;; all relevant data must be present
;xExist = n_elements(xData) gt 0
;yExist = n_elements(yData) gt 0
;qtyExist = n_elements(qty) gt 0
;errPresent = n_elements(err) gt 0
;if (~xExist || ~yExist || ~qtyExist) then return, 0
;
;; must be a 2D dataset
;ndim = size(qty,/n_dimensions)
;if (ndim ne 2) then return, 0
;
;nx = n_elements(xData)
;ny = n_elements(yData)
;nqty1 = (size(qty))[1]
;nqty2 = (size(qty))[2]
;if (nx ne nqty1) then begin
;   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (nx ne nqty1+1)) then begin
;      ;msg = 'Size of dependent and first independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endif
;endif
;
;;; If necessary, add a null value to xData to make it a 'false histogram'
;;; with one extra point relative to the dependent data.
;xIsPoint = strcmp(xType,'POINTS',4,/FOLD_CASE)
;if (xIsPoint) then xData = [xData,0.0]
;
;;; If necessary, add a null value to yData to make it a 'false histogram'
;;; with one extra point relative to the dependent data.
;yIsPoint = strcmp(yType,'POINTS',4,/FOLD_CASE)
;if (yIsPoint) then yData = [yData,0.0]
;
;if (ny ne nqty2) then begin
;   isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (ny ne nqty2+1)) then begin
;      ;msg = 'Size of dependent and second independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endif
;endif
;
;; Generate errors if not present
;if (~errPresent) then begin
;   index = where(qty le 0, cnt)
;   err = qty
;   if (cnt gt 0) then err[index] = 1.0
;   err = temporary(sqrt(err))
;endif
;
;xValue = temporary(xData)
;yValue = temporary(yData)
;dataValue = temporary(qty)
;errorValue = temporary(err)
;xTitle = xLabel
;yTitle = yLabel
;dataTitle = qtyLabel
;
;return, retVal
;end



;===============================================================================
; ASCIISPEDataset::SetProperty
; 
; PURPOSE:
;   
;
; PARAMETERS
;
; KEYWORDS:
;
;   axis1Value - the actual values of the first independent axis - a vector
;   
;   axis2Value - the actual values of the second independent axis - a vector
;   
;   dataValue - the actual values for the dependent data - a 2D array
;   
;   errorValue - the actual errors in the dependent data - a 2D array
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;
;pro ASCIISPEDataset::SetProperty, axis1Value=axis1Value,axis2Value=axis2Value,dataValue=dataValue,errorValue=errorValue $
;   ,_EXTRA=etc
;compile_opt idl2
;
;;
;if (n_elements(axis1Value) gt 0 && obj_valid(Self.axis1Ref)) then begin
;   void = (Self.axis1Ref)->SetData(axis1Value)
;endif 
;;
;if (n_elements(axis2Value) gt 0 && obj_valid(Self.axis2Ref)) then begin
;   void = (Self.axis2Ref)->SetData(axis2Value)
;endif 
;;
;if (n_elements(dataValue) gt 0 && obj_valid(Self.dataRef)) then begin
;   void = (Self.dataRef)->SetData(dataValue)
;endif 
;;
;if (n_elements(errorValue) gt 0  && obj_valid(Self.errorRef)) then begin
;   void = (Self.errorRef)->SetData(errorValue)
;endif 
;
;; call base class mutator
;if (n_elements(etc) gt 0) then Self->GenericDataset::SetProperty, _EXTRA=etc
;
;end


;===============================================================================
; ASCIISPEDataset::GetProperty
; 
; PURPOSE:
;   
;
; PARAMETERS
;
; KEYWORDS:
;   axis1Ref - the object reference to the first dependent axis of the dataset
;   
;   axis2Ref - the object reference to the second dependent axis of the dataset, if it is 2D
;   
;   dataRef - the object reference to the dependent data
;   
;   errorRef - the object reference to the error in the dependent data
;   
;   axis1Value - the actual values of the first independent axis - a vector
;   
;   axis2Value - the actual values of the second independent axis, if a 2D dataset - a vector
;   
;   dataValue - the actual values for the dependent data - either 1D or 2D array
;   
;   errorValue - the actual errors in the dependent data - either 1D or 2D array
;   
;   nDimensions - 1 | 2 if a 1D | 2D dataset
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;
;pro ASCIISPEDataset::GetProperty,axis1Ref=axis1Ref,axis2Ref=axis2Ref,dataRef=dataRef,errorRef=errorRef, plotDataRef=plotDataRef $
;   ,axis1Value=axis1Value,axis2Value=axis2Value,dataValue=dataValue,errorValue=errorValue, nDimensions=nDimensions $
;   ,axis1Label=axis1Label, axis2Label=axis2Label, dataLabel=dataLabel $
;   ,axis1Units=axis1Units, axis2Unitsl=axis2Units, dataUnits=dataUnits $
;   ,axis1Distribution=axis1Distribution, axis2Distribution=axis2Distribution $
;   ,_REF_EXTRA=etc
;compile_opt idl2
;
;;
;if (arg_present(instrument)) then begin
;   void = (Self.instrumentRef)->GetData(instrument)
;endif
;;
;if (arg_present(treatment)) then begin
;   void = (Self.treatmentRef)->GetData(treatment)
;endif
;;
;if (arg_present(axis1Label) || arg_present(axis1Units) || arg_present(axis1Distribution)) then begin
;   if ((void = (Self.axis1Ref)->GetMetaData('Long_name',axis1Label)) eq 0 ) then axis1Label=''
;   if ((void = (Self.axis1Ref)->GetMetaData('Units',axis1Units)) eq 0 ) then axis1Units=''
;   if ((void = (Self.axis1Ref)->GetMetaData('Distribution',axis1Distribution)) eq 0 ) then axis1Distribution='POINTS'
;endif
;;
;if (arg_present(axis2Label) || arg_present(axis2Units) || arg_present(axis2Distribution)) then begin
;   if ((void = (Self.axis2Ref)->GetMetaData('Long_name',axis2Label)) eq 0 ) then axis2Label=''
;   if ((void = (Self.axis2Ref)->GetMetaData('Units',axis2Units)) eq 0 ) then axis2Units=''
;   if ((void = (Self.axis2Ref)->GetMetaData('Distribution',axis2Distribution)) eq 0 ) then axis2Distribution='POINTS'
;endif
;;
;if (arg_present(dataLabel) || arg_present(dataUnits)) then begin
;   if ((void = (Self.dataRef)->GetMetaData('Long_name',dataLabel)) eq 0 ) then dataLabel=''
;   if ((void = (Self.dataRef)->GetMetaData('Units',dataUnits)) eq 0 ) then dataUnits=''
;endif
;;
;if (arg_present(axis1Value)) then begin
;   void = (Self.axis1Ref)->GetData(axis1Value)
;endif
;;
;if (arg_present(axis2Value) && (Self.ndims eq 2)) then begin
;   void = (Self.axis2Ref)->GetData(axis2Value)
;endif
;;
;if (arg_present(dataValue)) then begin
;   void = (Self.dataRef)->GetData(dataValue)
;endif
;;
;if (arg_present(errorValue)) then begin
;   void = (Self.errorRef)->GetData(errorValue)
;endif
;;
;axis1Ref = Self.axis1Ref
;;
;axis2Ref = Self.axis2Ref
;;
;dataRef = Self.dataRef
;;
;errorRef = Self.errorRef
;;
;plotDataRef=Self.plotDataRef
;;
;nDimensions = Self.ndims
;;
;
;
;; call base class accessor
;if (n_elements(etc) gt 0) then Self->GenericDataset::GetProperty, _EXTRA=etc
;
;end


;===============================================================================
; ASCIISPEDataset::Clone
; 
; PURPOSE:
;   return an object containing an axact copy of the data in the current object
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - a copy of the current object
;    failure - a null object
;
function ASCIISPEDataset::Clone
compile_opt idl2

status = Self->toASCIISPE(dataStr)

if (status) then begin
   filename = 'Copy of '+Self.filename
   Self->GetProperty, name=nameTag, treatment=trmt
   nameTag = 'Copy of '+nameTag
   oData = obj_new('ASCIISPEDataset',dataStr=dataStr,nameTag=nameTag)
   if (obj_valid(oData)) then oData->SetProperty, filename=filename, treatment=trmt
   return, oData
end

return, obj_new() 

end



;===============================================================================
; ASCIISPEDataset::Init
; 
; PURPOSE:
;   Initialization method for objects of ASCIISPEDataset class. For the initialization to 
;   complete successfully, at least a filename or the dataStr keyword must be supplied.
;
; PARAMETERS
;   filename   - The name of a dave format file from which to retrieve the dataset.
;
;   nameTag    - A label to be associated with objects of this class. If undefined, it will 
;                default to the basename of the filename keyword, if that is set, or to 'UNTITLED'
;
;   dataStr    - a data structure containing a SPE dataset This is an anonymous data structure
;                containing the following fields:
;                   {xData: float[n], $    ; a vector representing first independent axis eg energy transfer or time-of-flight, etc
;                    yData: float[m], $    ; a vector representing second independent axis eg angle, wavevector, etc
;                    data:  float[n,m], $  ; a 2D array of dependent data values
;                    error: float[n,m], $  ; a 2D array of the uncertainty in the dependent data values
;                    xTitle: ''       , $  ; title for xData block
;                    yTitle: ''       , $  ; title for yData block
;                    dataTitle: ''    , $  ; title for data block
;                   }
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function ASCIISPEDataset::Init, filename=filename, dataStr=dataStr, nameTag=nameTag, _REF_EXTRA=etc
compile_opt idl2

; Basic error handler
catch, theError
if (theError ne 0) then begin
   catch, /cancel
   return, 0
endif

filenameIsSet = n_elements(filename)
dataStrIsSet = n_elements(dataStr)
nameTagIsSet = n_elements(nameTag)

; Can't proceed if both filename and dataStr are undefined
if (~filenameIsSet && ~dataStrIsSet) then return, 0

; If starting from filename, read it in
if (filenameIsSet and ~dataStrIsSet) then begin
   dm_load_spe,filename,zdat=dat,zerr=err,xdat=ydat,ydat=xdat,xtit=ytit,ytit=xtit,ztit=ztit,$
       error=error,xhist=yhist,yhist=xhist  ; note that x and y axis are swapped in the call!

   if (error ne 0) then return, 0
   
   dat = temporary(transpose(dat))
   err = temporary(transpose(err))
   
   xMode = (xhist eq 0)? 'POINT' : 'HISTOGRAM' 
   yMode = (yhist eq 0)? 'POINT' : 'HISTOGRAM'
      
   xName = (strtrim(xtit) eq '')? 'X' : xtit
   yName = (strtrim(ytit) eq '')? 'Y' : ytit
   datName = (strtrim(ztit) eq '')? 'Intensity' : ztit
   errName = 'Error'
   
   xDesc = xName
   yDesc = yName
   zDesc = datName
   
   i = strpos(filename,'.',/reverse_search)
   ext = (i lt 0)? '' : strmid(filename,i)
   ; strip dir path and extension from the filename and assign to nameTag, if required
   basename = file_basename(filename,ext,/fold_case)
   if (~nameTagisSet) then nameTag = basename
endif else if (dataStrIsSet) then begin
   xdat = dataStr.xData
   ydat = dataStr.yData
   dat = dataStr.data
   err = dataStr.error
   xtit = dataStr.xTitle
   ytit = dataStr.yTitle
   ztit = dataStr.dataTitle
   
   zdim = size(dat)
   d1 = zdim[1]
   d2 = zdim[2]
   nx = n_elements(xdat)
   ny = n_elements(ydat)

   xMode = (nx eq d1)? 'POINT' : 'HISTOGRAM' 
   yMode = (ny eq d2)? 'POINT' : 'HISTOGRAM'

   xName = (strtrim(xtit) eq '')? 'X' : xtit
   yName = (strtrim(ytit) eq '')? 'Y' : ytit
   datName = (strtrim(ztit) eq '')? 'Intensity' : ztit
   errName = 'Error'
   
   xDesc = xName
   yDesc = yName
   zDesc = datName
endif

; Check for masked data points 
; For spe files, masked points have the value -1e20
; Replace masked points with a NaN
index = where(dat eq -1e20, nMasked)
if (nMasked gt 0) then dat[index] = !values.F_NAN

; If nameTag is not yet specified, set it to 'Untitled'
if (n_elements(nameTag) eq 0) then nameTag = 'UNTITLED'
description = (filenameIsSet)? filename : nameTag
basename = (filenameIsSet)? basename : nameTag

; iTools does not handle '_[0-9]' characters in nameTags well eg _0, _1, etc
; replace all '_' with '-'
;while ((pos = stregex(nameTag,'_[0-9]')) ne -1) do begin
;    strput, nameTag,'-',pos
;endwhile
   

; Call the base class init method to instantiate container
status = self->GenericDataset::Init(name=nameTag $
                ,description=description,identifier='id'+file_basename(nameTag),icon='surface' $
                ,type='ASCIISPE', _EXTRA=etc)
if (status eq 0) then return, status

Self->AddMetaData,'DatasetName',basename ; give it same name as the file basename

xAxisType=0
yAxisType=1
datAxisType=2
errAxisType=3
oX = obj_new('IDLitDataDAVE', xdat, name=xName,type=partype(xdat), description=xDesc, axisType=xAxisType)
oY = obj_new('IDLitDataDAVE', ydat, name=yName,type=partype(ydat), description=yDesc, axisType=yAxisType)
oDat = obj_new('IDLitDataDAVE',dat,name=datName,type=partype(dat), description=zDesc, axisType=datAxisType)
oErr = obj_new('IDLitDataDAVE',err,name=errName,type=partype(err), description=errName, axisType=errAxisType)

;; metadata
oX->AddMetaData,'Long_name',xDesc
oX->AddMetaData,'Units',''
oX->AddMetaData,'Distribution',xMode

oY->AddMetaData,'Long_name',yDesc
oY->AddMetaData,'Units',''
oY->AddMetaData,'Distribution',yMode

oDat->AddMetaData,'Signal',1
oDat->AddMetaData,'Axes',[xName,yName]
oDat->AddMetaData,'Long_name',zDesc
oDat->AddMetaData,'Units',''

Self->Add, [oX,oY,oDat,oErr] $
            ,parameter_name=strtrim(string([xAxisType,yAxisType,datAxisType,errAxisType]),2)
            
; Add a blank treatment history object
child = obj_new('IDLitData','',name='Treatment History',type='HISTORY' $
                ,description='Treatment history')
if (obj_valid(child)) then begin
   Self->add, child
   Self.treatmentRef = child
endif

Self.axis1Ref = oX
Self.axis2Ref = oY
Self.dataRef = oDat
Self.errorRef = oErr
;self.plotDataRef = Self
Self.filename = description
Self.nDims = 2

return, 1
end


;===============================================================================
pro ASCIISPEDataset__Define
compile_opt idl2

void = {ASCIISPEDataset $
        ,inherits GenericDataset $
        }

end