; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.


;----------------------------------------------------------------------------
; MARK: write_unix_launcher
;
FUNCTION write_unix_launcher, p_distinfo

  saveFileBasename = FILE_BASENAME((*p_distinfo).s_file)
  idlDirBasename = FILE_BASENAME((*p_distinfo).idldir)
  rtMode = (*p_distinfo).rtMode
  
  IF saveFileBasename ne '' THEN BEGIN
     runfile = './'+saveFileBasename
  ENDIF ELSE BEGIN
     runfile = ''
  ENDELSE

  launcherTxt = [ $
    '#!/bin/sh', $
    '#', $
    '# This script starts an IDL Runtime or Virtual Machine application', $
    '# from an IDL installation located in a subdirectory of the directory', $
    '# containing the script.', $
    '#', $
    '', $
    '', $
    '# Find the location of this script', $
    'topdir=`dirname $0`', $
    'if (test $topdir = ".") ; then', $
    '   topdir=$PWD;', $
    'fi', $
    '', $
    '# Specify the path to the IDL SAVE file that launches', $
    '# the application, relative to $topdir.', $
    'idlapp=' + runfile, $
    '', $
    '# Specify the path to the top directory of the IDL', $
    '# distribution, relative to $topdir.', $
    'idl_install_dir=' + idlDirBasename, $
    'IDL_DIR=$topdir/$idl_install_dir ; export IDL_DIR', $
    '', $
    '# Change the working directory', $
    'cd $topdir', $
    '', $
    '# Run the application', $
    'exec $IDL_DIR/bin/idl -' + rtMode + '=$idlapp' $
  ]
  
  ; write the text of the launch script
  launchScript = (*p_distinfo).outpath + PATH_SEP() + $
    (*p_distinfo).appname
  OPENW, startout, launchScript, /GET_LUN
  for i=0, n_elements(launcherTxt)-1 do begin
     PRINTF, startout, launcherTxt[i]
  endfor
  FREE_LUN, startout
  FILE_CHMOD, launchScript, '755'o

  RETURN, 'Wrote UNIX launcher'
END


;----------------------------------------------------------------------------
; MARK: write_mac_launcher
;
FUNCTION write_mac_launcher, p_distinfo
  saveFileBasename = FILE_BASENAME((*p_distinfo).s_file)
  idlDirBasename = FILE_BASENAME((*p_distinfo).idldir)
  rtMode = (*p_distinfo).rtMode
  
  IF saveFileBasename ne '' THEN BEGIN
     runfile = saveFileBasename
  ENDIF ELSE BEGIN
     runfile = ''
  ENDELSE

  launcherTxt = [ $
    '(*', $
    'This script creates a "double-clickable" icon for a runtime IDL', $
    'application defined by the idlApp variable. This script should be placed', $
    'at the top level of a runtime application hierarchy. The ', $
    'Utils_applescripts.scpt file must be in the same directory.', $
    '*)', $
    '', $
    '(*', $
    'Specify the path to the IDL SAVE file that launches the application, ', $
    'relative to the location of the script', $
    '*)', $
    '', $
    'set idlApp to "' + runfile + '" as string', $
    '', $
    '(*', $
    'Specify the path to the top directory of the IDL distribution, ', $
    'relative to the location of the script.', $
    '*)', $
    'set idlDir to "' + idlDirBasename + '" as string', $
    '', $
    'tell application "Finder"', $
    '  set myContainer to (container of (path to me)) as string', $
    '  set IDLDirFolder to POSIX path of myContainer & idlDir & "/"', $
    '  set IDLRunFolder to quoted form of (IDLDirFolder & "bin")', $
    '  set ApplescriptUtilsFile to myContainer & "Utils_applescripts.scpt" as string', $
    'end tell', $
    '', $
    'set myAppPath to POSIX path of myContainer & "/" & idlApp as string', $
    'if idlApp is equal to "" then', $
    '   set myAppPath to "" ', $
    'end if', $
    '', $
    'set idlCmd to IDLDirFolder & "bin/idl -' + rtMode + '=" & myAppPath', $
    '', $
    'set ApplescriptUtils to load script file ApplescriptUtilsFile', $
    'tell ApplescriptUtils', $
    '  set XResult to LaunchX11()', $
    '  EnvironmentSetup(IDLDirFolder)', $
    'end tell', $
    '', $
    'if XResult is equal to 0 then', $
    '  set theCommand to shellCmd & "''" & fullSetupCmd & "; " & DisplayCmd & "; cd  " & IDLRunFolder & "; " & idlCmd & "'' > /dev/null  2>&1 & "', $
    '  --display dialog theCommand', $
    '  set results to do shell script theCommand', $
    'end if' $
  ]


  ; write the text for the launch script
  ; this will need to be converted to applescript
  launchTextFilename = (*p_distinfo).outpath + PATH_SEP() + $
    (*p_distinfo).appname + '_mac_script_source.txt'
  OPENW, startout, launchTextFilename, /GET_LUN
  for i=0, n_elements(launcherTxt)-1 do begin
     PRINTF, startout, launcherTxt[i]
  endfor
  FREE_LUN, startout

  ; convert the text version of the script to AppleScript
  launchAplScriptname = (*p_distinfo).outpath + PATH_SEP() + $
    (*p_distinfo).appname + '.app'
  compileCommand = 'osacompile -o ' + launchAplScriptname + ' ' + $
    launchTextFilename
  if !version.os EQ 'darwin' then begin
    spawn, compileCommand, result
  endif else begin
    md_log, 'Not running on a mac could not convert text version '+$
            'of the script to AppleScript', (*p_distinfo).logfile
  endelse



  ; Just copy the utilities script
  IF FILE_TEST((*p_distinfo).mk_rt_dir + 'Utils_applescripts.scpt', /READ) THEN BEGIN
    FILE_COPY, (*p_distinfo).mk_rt_dir + 'Utils_applescripts.scpt', $
      (*p_distinfo).outpath, /OVERWRITE
  ENDIF ELSE BEGIN
    md_log, 'Utils_applescripts.scpt not found or not readable', (*p_distinfo).logfile
  ENDELSE
  
  RETURN, 'Wrote Macintosh launcher'
  

END


;----------------------------------------------------------------------------
; MARK: write_win_autorun
;
FUNCTION write_win_autorun, p_distinfo

  appname = (*p_distinfo).appname

  launcherTxt = [ $
    '[autorun]', $
    'open = ' + appname + '.exe', $
    'icon= idl.ico' $
  ]

  ; write the text for the autorun.inf file
  OPENW, startout, (*p_distinfo).outpath + path_sep() + 'autorun.inf', /GET_LUN
  for i=0, n_elements(launcherTxt)-1 do begin
     PRINTF, startout, launcherTxt[i]
  endfor
  FREE_LUN, startout

  RETURN, 'Wrote Windows_autorun'
END


;----------------------------------------------------------------------------
; MARK: write_win_ini
;
FUNCTION write_win_ini, p_distinfo

  saveFileBasename = FILE_BASENAME((*p_distinfo).s_file)
  idlDirBasename = FILE_BASENAME((*p_distinfo).idldir)

  filename = (*p_distinfo).outpath + path_sep() + (*p_distinfo).appname + '.ini'
  if file_test((*p_distinfo).mk_rt_dir + 'start_app_win.ini', /read) then begin
    file_copy, (*p_distinfo).mk_rt_dir + 'start_app_win.ini', filename, /overwrite
  endif else begin
    md_log, (*p_distinfo).mk_rt_dir + $
      'start_app_win.ini not found or not readable', (*p_distinfo).logfile
    return, 'error writing Windows_ini'
  endelse

  iniTxt = strarr(file_lines(filename))
  openr, lun, filename, /get_lun
  readf, lun, iniTxt
  free_lun, lun
  commandSuffix =  ` -${(*p_distinfo).rtMode}=${saveFileBasename}`
  foreach s, iniTxt, i do begin
    if (s.Contains('Action') && s.Contains('<IDL_DIR>')) then begin
      iniTxt[i] = s.Replace('<IDL_DIR>', '.\' + idlDirBasename) + commandSuffix
    endif else if (s.Contains('My App')) then begin
      iniTxt[i] = s.Replace('My App', ((*p_distinfo).appname).CapWords())
    endif
  endforeach
  openw, lun, filename, /get_lun
  printf, lun, transpose(iniTxt)
  free_lun, lun

  return, 'Wrote Windows_ini'
END


;----------------------------------------------------------------------------
; MARK: write_win_launcher
;
FUNCTION write_win_launcher, p_distinfo

  ; Strings that will be replaced in the generic .ini file
  
  ; Copy static files
  IF FILE_TEST((*p_distinfo).mk_rt_dir + 'idl.ico', /READ) THEN BEGIN
    FILE_COPY, (*p_distinfo).mk_rt_dir + 'idl.ico', (*p_distinfo).outpath, $
      /OVERWRITE
  ENDIF ELSE BEGIN
    md_log, (*p_distinfo).mk_rt_dir + 'idl.ico not found or not readable', $
      (*p_distinfo).logfile
  ENDELSE
  
  IF FILE_TEST((*p_distinfo).mk_rt_dir + 'splash.bmp', /READ) THEN BEGIN
    FILE_COPY, (*p_distinfo).mk_rt_dir + 'splash.bmp', (*p_distinfo).outpath, $
      /OVERWRITE
  ENDIF ELSE BEGIN
    md_log, (*p_distinfo).mk_rt_dir + 'splash.bmp not found or not readable', $
      (*p_distinfo).logfile
  ENDELSE
  
  IF FILE_TEST((*p_distinfo).mk_rt_dir + 'start_app_win.exe', /READ) THEN BEGIN
    FILE_COPY, (*p_distinfo).mk_rt_dir + 'start_app_win.exe', $
      (*p_distinfo).outpath + PATH_SEP() + (*p_distinfo).appname + '.exe', $
      /OVERWRITE
  ENDIF ELSE BEGIN
    md_log, (*p_distinfo).mk_rt_dir + $
      'start_app_win.exe not found or not readable', (*p_distinfo).logfile
  ENDELSE
  
  ; Write autorun.inf
  ; md_log, write_win_autorun(p_distinfo), (*p_distinfo).logfile
  
  ; Write start_app_win.ini
  md_log, write_win_ini(p_distinfo), (*p_distinfo).logfile
  
  RETURN, 'Wrote windows launcher'
END


;----------------------------------------------------------------------------
; MARK: copy_launcher
;
FUNCTION copy_launcher, p_distinfo

  IF (*p_distinfo).win64 THEN $
    md_log, write_win_launcher(p_distinfo), (*p_distinfo).logfile
  IF (*p_distinfo).mac THEN $
    md_log, write_mac_launcher(p_distinfo), (*p_distinfo).logfile
  IF (*p_distinfo).unix THEN $
    md_log, write_unix_launcher(p_distinfo), (*p_distinfo).logfile
    
  RETURN, 'Finished writing launchers'
END


;----------------------------------------------------------------------------
; MARK: copy_savefile
;
FUNCTION copy_savefile, p_distinfo

  FILE_COPY, (*p_distinfo).s_file, (*p_distinfo).outpath, /OVERWRITE
  
  RETURN, 'Copied savefile'
  
END


;----------------------------------------------------------------------------
; MARK: copy_licenseutils
;
FUNCTION copy_licenseutils, p_distinfo

  ; Create the license_utils directory and copy all contents
  ; This is done here as a requirement rather than with the manifest_rt.txt file.
  ; The platform specific file patterns also prevent copy_manifest from copying
  ; the 64-bit binaries needed even for a Windows 32-bit distribution. (We don't
  ; ship 32-bit activate.exe.)
  idlDirBasename = FILE_BASENAME((*p_distinfo).idldir)
  src_license_utils = (*p_distinfo).idldir + PATH_SEP() + "license_utils"
  dest_license_utils = (*p_distinfo).outpath + PATH_SEP() + idlDirBasename
  FILE_MKDIR, dest_license_utils
  FILE_COPY, src_license_utils, dest_license_utils, /OVERWRITE, /RECURSIVE

  RETURN, 'Created license directory and copied license_utils'
END


;----------------------------------------------------------------------------
; MARK: copy_manifest
;
FUNCTION copy_manifest, p_distinfo, $
   VERBOSE=verbose
  compile_opt idl2, logical_predicate
    
  ; Add a loading bar so IDL doesnt appear to hang.
  ; Old behavior is now accomplished with the verbose keyword.
  ; IDL-71186
  if verbose then begin
    print, 'Determining which files to copy...'
  endif else if (~!quiet) then begin
    ; At this point we have little to no idea how big this thing is actually 
    ; going to be so we determine a end point and update by percentages 
    ; normalized by a recorded average time.
    CLI_PROGRESS.INITIALIZE,TEXT = 'Collecting files', MAXIMUM = 1000
    
    ; counter lets us keep track of length over multiple for loops.
    counter = 0.0
    CLI_PROGRESS.UPDATE, counter
  endif
  
  ; Read the manifest file
  
  md_log, 'Looking in manifest file '+(*p_distinfo).m_file, (*p_distinfo).logfile
  nlines = FILE_LINES((*p_distinfo).m_file)
  array = STRARR(nlines)
  OPENR, unit, (*p_distinfo).m_file, /GET_LUN
  READF, unit, array
  FREE_LUN, unit

  ; Create a list of manifest file lines
  
  lines = LIST(array, /EXTRACT)

  ; Clean up the list of manifest lines
  ;
  ; (1) Trim off blanks on both ends
  ; (2) Remove blank lines
  ; (3) Remove commented-out lines
  ; (4) Trim off comments

  indicesToRemove = LIST()
  
  FOR i=0, N_ELEMENTS(lines)-1 DO BEGIN
    
    lines[i] = STRTRIM(lines[i], 2)
    
    IF lines[i] EQ '' THEN BEGIN
      indicesToRemove.Add, i
      CONTINUE
    ENDIF

    semi = STRPOS(lines[i], ';')
    IF semi EQ 0 THEN BEGIN
      indicesToRemove.Add, i
      CONTINUE
    ENDIF ELSE IF semi NE -1 THEN BEGIN
      lines[i] = STRTRIM(STRMID(lines[i], 0, semi), 2)
    ENDIF

    IF lines[i].StartsWith('./') THEN BEGIN
      lines[i] = lines[i].Remove(0, 1)
    ENDIF

  ENDFOR

  lines.Remove, indicesToRemove.ToArray()
  
  ; Change the manifest based on keywords
  ;
  ; Note that many of these assume the <bin.platform.arch> directories are
  ; already explicitly additive in the manifest file.
  
  IF ~(*p_distinfo).win64 THEN BEGIN
    lines.Add, '[-] bin/bin.x86_64/'
  ENDIF
  
  IF (*p_distinfo).dataminer THEN BEGIN
    lines.Add, '*odbc*'
    lines.Add, 'resource/dm/'
  ENDIF ELSE BEGIN
    lines.Add, '[-] *odbc*'
    lines.Add, '[-] bin/bin.x86_64/idl_dataminer.*'
    lines.Add, '[-] bin/bin.x86_64/dm/'
    lines.Add, '[-] bin/bin.linux.x86_64/idl_dataminer.*'
    lines.Add, '[-] bin/bin.linux.x86_64/dm/'
    lines.Add, '[-] bin/bin.darwin.x86_64/idl_dataminer.*'
    lines.Add, '[-] bin/bin.darwin.arm64/idl_dataminer.*'
    lines.Add, '[-] resource/dm/'
  ENDELSE

  IF ~(*p_distinfo).dicomex THEN BEGIN
    lines.Add, '[-] bin/bin.x86_64/idl_dicomex*'
    lines.Add, '[-] bin/bin.linux.x86_64/idl_dicomex*'
    lines.Add, '[-] bin/bin.darwin.x86_64/idl_dicomex*'
  ENDIF
  
  IF (*p_distinfo).grib THEN BEGIN
    lines.Add, 'resource/grib/'
  ENDIF ELSE BEGIN
    lines.Add, '[-] bin/bin.x86_64/idl_grib*'
    lines.Add, '[-] bin/bin.linux.x86_64/idl_grib*'
    lines.Add, '[-] bin/bin.darwin.x86_64/idl_grib*'
    lines.Add, '[-] bin/bin.darwin.arm64/idl_grib*'
    lines.Add, '[-] resource/grib/'
  ENDELSE
  
  IF ~(*p_distinfo).hires_maps THEN BEGIN
    lines.Add, '[-] resource/maps/high/'
  ENDIF
  
  IF ~(*p_distinfo).hires_shapefiles THEN BEGIN
    lines.Add, '[-] resource/maps/shape/GSHHS_h*'
    lines.Add, '[-] resource/maps/shape/states_high*'
  ENDIF
 
  if ~(*p_distinfo).python then begin
    python = (file_search((*p_distinfo).idldir, 'idl-python*'))[0]
    if (strlen(python) gt 0) then begin
      base = file_basename(python)
      lines.Add, '[-] bin/bin.x86_64/' + base
      lines.Add, '[-] bin/bin.linux.x86_64/' + base
      lines.Add, '[-] bin/bin.darwin.x86_64/' + base
      lines.Add, '[-] bin/bin.darwin.arm64/' + base
    endif
  endif
  
  if ~(*p_distinfo).browser then begin
    lines.Add, '[-] bin/bin.x86_64/browser'
    lines.Add, '[-] bin/bin.x86_64/libcef.dll'
    lines.Add, '[-] bin/bin.linux.x86_64/browser'
    lines.Add, '[-] bin/bin.linux.x86_64/libcef.so'
    lines.Add, '[-] bin/bin.darwin.x86_64/IDL Browser Engine.app'
    lines.Add, '[-] bin/bin.darwin.arm64/IDL Browser Engine.app'
  endif

  IF ~(*p_distinfo).video THEN BEGIN
    
    directories = ['bin.x86_64', 'bin.linux.x86_64', 'bin.darwin.x86_64', 'bin.darwin.arm64']
    files = ['idl_video*', '*avcodec*', '*avformat*', '*avfilter*', '*avutil*', '*swscale*']
    
    FOREACH directory, directories DO BEGIN
      FOREACH file, files DO BEGIN
        lines.Add, '[-] bin/' + directory + '/' + file
      ENDFOREACH
    ENDFOREACH

  ENDIF
  
  ; Move the "negations" to a separate list
  
  indicesToRemove = LIST()
  
  FOREACH line, lines, i DO BEGIN
    IF line.StartsWith('[-]') THEN BEGIN
      indicesToRemove.Add, i
      lines[i] = STRTRIM(line.Remove(0,2), 1)
    ENDIF
    
  ENDFOREACH
  
  negations = lines.Remove(indicesToRemove.ToArray())
  
  ; Collect individual source files
  ;
  ; All manifest lines that specify a directory are expanded into the "regular"
  ; files they contain.  These files are put into a 'sources' list.  That list
  ; will not contain any directories.

  sources = list()

  FOREACH line, lines DO BEGIN

    source = (*p_distinfo).idldir + PATH_SEP() + line

    IF FILE_TEST(source, /DIRECTORY) THEN BEGIN
      files = FILE_SEARCH(source, '*', /MATCH_INITIAL_DOT, /TEST_REGULAR)
      IF ISA(files, /array) THEN BEGIN
        sources.Add, files, /extract
      ENDIF
    ENDIF ELSE BEGIN
      files = FILE_SEARCH(source, /MATCH_INITIAL_DOT, /TEST_REGULAR, COUNT = n)  ; wildcards
      IF n GT 0 THEN BEGIN
        sources.Add, files, /extract
      ENDIF
    ENDELSE
    
    if (~verbose && ~!quiet) then begin
      CLI_PROGRESS.UPDATE, counter
        ; 60.0 / n_elements(lines) is 
        ; 60 / 1000 % (6%). This 6% is found from 5.8s / 95.6s
        ; (time for this section / time for whole section)
      counter += 60.0 / n_elements(lines)
    endif
    
  ENDFOREACH

  ; Remove negations from source collection

  if (~verbose && ~!quiet) then begin
    CLI_PROGRESS.TEXT = "Filtering files"
  endif

  sources = sources.ToArray()  ; removing from an array is faster than from list
  
  FOREACH line, negations DO BEGIN
    
    target = (*p_distinfo).idldir + PATH_SEP() + line

    IF FILE_TEST(target, /DIRECTORY) THEN BEGIN
      files = FILE_SEARCH(target, '*', /MATCH_INITIAL_DOT, /TEST_REGULAR)
    ENDIF ELSE BEGIN
      files = FILE_SEARCH(target, /MATCH_INITIAL_DOT, /TEST_REGULAR, COUNT = n)  ; wildcards
      IF n EQ 0 THEN BEGIN
        files = !NULL
      ENDIF
    ENDELSE
    
    ; Change to to this loop at CL:473823
    ; The change batches this process to speed it up.
    mask = bytarr(n_elements(sources)) + 1b
    foreach f, files do begin
      mask and= (sources ne f) 
    endforeach
    sources = sources[where(mask)]
    
    if (~verbose && ~!quiet) then begin
      CLI_PROGRESS.UPDATE, counter
      counter += 150.0 / n_elements(negations)
    endif
    
  ENDFOREACH
  
  ; Copy the files

  if (~verbose && ~!quiet) then begin
    CLI_PROGRESS.TEXT = "Copying files"
  endif

  n = STRLEN((*p_distinfo).idldir)  ; length of base source directory string

  FOREACH source_file, sources DO BEGIN
    
    output_file = (*p_distinfo).outpath + PATH_SEP() + $
      FILE_BASENAME((*p_distinfo).idldir) + STRMID(source_file, n)

    IF FILE_TEST(source_file, /READ) THEN BEGIN
      IF ~FILE_TEST(FILE_DIRNAME(output_file), /DIRECTORY) THEN BEGIN
        FILE_MKDIR, FILE_DIRNAME(output_file)
      ENDIF
      ; Test for symlink and remove if necessary (49828)
      IF !version.os_family eq 'unix' && FILE_TEST(output_file, /SYMLINK) THEN $
        FILE_DELETE, output_file
        FILE_COPY, source_file, output_file, /OVERWRITE, /COPY_SYMLINK
      md_log, 'Copying: '+source_file, (*p_distinfo).logfile
      if ~verbose then begin
        if (~!quiet) then begin
          CLI_PROGRESS.UPDATE, counter
          counter += 790.0 / n_elements(sources)
        endif
      endif else begin
        print, 'Copying: '+source_file
      endelse
    ENDIF ELSE BEGIN
      md_log, 'Warning: Source file not found: ' + source_file, (*p_distinfo).logfile
    ENDELSE
          
  ENDFOREACH

  if (~verbose && ~!quiet) then begin
    CLI_PROGRESS.Finish
  endif
  
  RETURN, 'finished processing manifest'

END


;----------------------------------------------------------------------------
; MARK: check_current_platform
;
PRO check_current_platform, p_distinfo

  ; If no OS keywords were set, select the current OS
  IF ((*p_distinfo).win64 + $
    (*p_distinfo).macint64 + (*p_distinfo).macarm64 + $
    (*p_distinfo).lin64 EQ 0) THEN BEGIN
    CASE !version.os OF
      'Win32': BEGIN
        IF !version.memory_bits EQ 64 THEN (*p_distinfo).win64 = 1
      END
      'darwin': BEGIN
        IF !version.ARCH EQ 'x86_64' THEN (*p_distinfo).macint64 = 1
        IF !version.ARCH EQ 'arm64' THEN (*p_distinfo).macarm64 = 1
      END
      'linux': BEGIN
        IF !version.memory_bits EQ 64 THEN (*p_distinfo).lin64 = 1
      END
    ENDCASE
  ENDIF
  
  ; Check to see if *any* non-Macintosh Unix platform is selected
  (*p_distinfo).unix = (*p_distinfo).lin64
    
  ; Check to see if *any* Macintosh platform is selected
  (*p_distinfo).mac = (*p_distinfo).macint64 || (*p_distinfo).macarm64 
    
END


;----------------------------------------------------------------------------
; MARK: md_log
;
PRO md_log, msg, logfile

  OPENW, log, logfile, /GET_LUN, /APPEND
  PRINTF, log, msg
  FREE_LUN, log
  
END


;----------------------------------------------------------------------------
; MARK: md_error
;
PRO md_error, msg

  PRINT, '-----------------------------------------------'
  MESSAGE, 'Error making distribution:', LEVEL=-1, /CONTINUE
  PRINT, ''
  FOR i = 0L, N_ELEMENTS(msg)-1 DO BEGIN
    PRINT, msg[i]
  ENDFOR
  PRINT, ''
  PRINT, 'make_rt exiting...'
  PRINT, '-----------------------------------------------'
END


;----------------------------------------------------------------------------
; MARK: make_rt
;
PRO make_rt, appname, outdir, $
    SAVEFILE=s_file, $
    MANIFEST=m_file, $
    IDLDIR=idldir, $
    LOGFILE=logfile, $
    VM=vm, $ 
    WIN64=win64, $
    MACINT64=macint64, $
    MACARM64=macarm64, $
    LIN64=lin64, $
    OVERWRITE=overwrite, $
    VERBOSE=verbose, $
    
    ; feature keywords
    BROWSER=browser, $
    DATAMINER=dataminer, $
    DICOMEX=dicomex, $
    HIRES_MAPS=hires_maps, $
    HIRES_SHAPEFILES=hires_shapefiles, $
    PYTHON=python, $
    VIDEO=video, $
    GRIB=grib, $
    
    ; deprecated keywords
    EMBEDDED=embedded, $
    IDL_HELP=idl_help, $     ; deprecated in IDL 7.0
    IDL_ASSISTANT=idl_assistant, $  ; deprecated in IDL 8.0

    ; deprecated platform keywords
    LIN32=lin32, $
    MACINT32=macint32, $
    MACPPC32=macppc32, $
    SUN32=sun32, $
    SUN64=sun64, $
    SUNX86_64=sunx86_64, $
    WIN32=win32
  
  ; Issue an error and exit on deprecated keywords
  obsolete_keyword = !NULL
  obsolete_message = !NULL
  
  IF (N_ELEMENTS(embedded) EQ 1) THEN BEGIN
    obsolete_keyword = 'EMBEDDED'
  ENDIF ELSE IF (N_ELEMENTS(idl_help) EQ 1) THEN BEGIN
    obsolete_keyword = 'IDL_HELP'
  ENDIF ELSE IF (N_ELEMENTS(idl_assistant) EQ 1) THEN BEGIN
    obsolete_keyword = 'IDL_ASSISTANT'
  ENDIF ELSE IF (N_ELEMENTS(lin32) EQ 1) THEN BEGIN
    obsolete_keyword = 'LIN32'
  ENDIF ELSE IF (N_ELEMENTS(macint32) EQ 1) THEN BEGIN
    obsolete_keyword = 'MACINT32'
  ENDIF ELSE IF (N_ELEMENTS(macppc32) EQ 1) THEN BEGIN
    obsolete_keyword = 'MACPPC32'
  ENDIF ELSE IF (N_ELEMENTS(sun32) EQ 1) THEN BEGIN
    obsolete_keyword = 'SUN32'
  ENDIF ELSE IF (N_ELEMENTS(sun64) EQ 1) THEN BEGIN
    obsolete_keyword = 'SUN64'
  ENDIF ELSE IF (N_ELEMENTS(sunx86_64) EQ 1) THEN BEGIN
    obsolete_keyword = 'SUNX86_64'
  ENDIF ELSE IF (N_ELEMENTS(win32) EQ 1) THEN BEGIN
    obsolete_keyword = 'WIN32'
  ENDIF
  
  IF obsolete_keyword THEN BEGIN
    obsolete_message = 'WARNING: ' + obsolete_keyword + ' keyword is obsolete.'
    print, obsolete_message
  ENDIF

  ; Must have an application name
  IF (N_ELEMENTS(appname) EQ 0) THEN BEGIN
    md_error, 'You must specify an application name.'
    RETURN
  ENDIF
  
  ; Save file must be readable if specified
  IF KEYWORD_SET(s_file) && (FILE_TEST(s_file, /READ) NE 1) THEN BEGIN
    md_error, ['The Save file you specified', ' ', $
      '   '+s_file, ' ', 'is not readable or does not exist']
    RETURN
  ENDIF
  
  ; If no Save file specified, use null string
  IF KEYWORD_SET(s_file) EQ 0 THEN s_file=''
  
  ; Output dir must be specified
  IF (N_ELEMENTS(outdir) EQ 0) THEN BEGIN
    md_error, 'You must specify an output directory.'
    RETURN
  ENDIF
  
  ; Output dir must be writable
  IF (FILE_TEST(outdir, /DIRECTORY, /WRITE) NE 1) THEN BEGIN
    md_error, ['The output directory you specified', ' ', $
      '   '+outdir, ' ', 'is not writable or does not exist']
    RETURN
  ENDIF
  
  ; Check to see if dir is empty
  files = FILE_SEARCH(FILE_SEARCH(outdir, /MARK_DIRECTORY)+'*', count=cnt)
  IF ((cnt NE 0) && ~KEYWORD_SET(overwrite)) THEN BEGIN
    md_error, ['The output directory is not empty: '+outdir, $
               'Use the OVERWRITE keyword to write new ' + $
               'files into an existing directory.']
    RETURN
  ENDIF

  ; If no manifest file specified, use the one in bin/make_rt
  IF N_ELEMENTS(m_file) EQ 0 THEN $
    m_file = FILEPATH('manifest_rt.txt', $
    SUBDIR=['bin', 'make_rt'])
    
  ; Make sure the manifest keyword is set to a string
  IF (SIZE(m_file, /TNAME) NE 'STRING') THEN BEGIN
    md_error, 'You must specify a filename for the MANIFEST keyword'
    RETURN
  ENDIF
  
  ; Manifest file must be readable
  IF (FILE_TEST(m_file, /READ) EQ 0) THEN BEGIN
    md_error, ['The manifest file you specified', ' ', $
      '   '+STRING(m_file), ' ', 'is not readable or does not exit']
    RETURN
  ENDIF
  
  ; If IDLDIR is not set, use !DIR. Note that we don't check
  ; whether the specified IDLDIR is readable or even exists;
  ; if there is a problem, the errors go to the log file when
  ; the copies are attempted.
  IF KEYWORD_SET(idldir) EQ 0 THEN idldir=!DIR
  ; Get the actual file case.
  idldir = (File_Search(idldir))[0]
  
  ; Assume we'll use the -rt flag, but allow user to specify
  ; the app *must* run in the Virtual Machine or with an embedded
  ; license.
  rtMode = 'rt'
  IF KEYWORD_SET(vm) THEN rtMode = 'vm'
  IF ~KEYWORD_SET(s_file) THEN rtMode = 'vm'
  
  ; Dataminer files are not copied unless DATAMINER=1
  dataminer = (KEYWORD_SET(dataminer) NE 0 && dataminer EQ 1) ? 1 : 0

  ; DICOMEX files are not copied unless DICOMEX=1
  dicomex = (KEYWORD_SET(dicomex) NE 0 && dicomex EQ 1) ? 1 : 0

  ; IDL_Assistant executables no longer exist in 8.0 and above
  ; Always set to 0 regardless of keyword
  idl_assistant = 0

  ; IDL Help system files are not copied unless IDL_HELP=1
  ; IDL_HELP not supported in IDL 7.0, but leave the infrastructure
  idl_help = (KEYWORD_SET(idl_help) NE 0 && idl_help EQ 1) ? 1 : 0

  ; High-res maps are not copied unless HIRES_MAPS=1
  hires_maps = (KEYWORD_SET(hires_maps) NE 0 && hires_maps EQ 1) ? 1 : 0

  ; High-res shape maps are not copied unless HIRES_SHAPEFILES
  hires_shapefiles = (KEYWORD_SET(hires_shapefiles) NE 0 $
    && hires_shapefiles EQ 1) ? 1 : 0

  ; Verbose. This keyword makes it so we print each copy on seperate lines.
  ; This was the standard behavior in version 9.1 and earlier
  verbose = (KEYWORD_SET(verbose) ? 1 : 0)

  ; Video
  video = (KEYWORD_SET(video) NE 0 && video EQ 1) ? 1 : 0

  ; GRIB
  grib = (KEYWORD_SET(grib) NE 0 && grib EQ 1) ? 1 : 0
  
  ; Create the output directory
  outpath = outdir + PATH_SEP() + appname
  IF (FILE_TEST(outpath, /DIRECTORY, /WRITE) NE 1) THEN $
    FILE_MKDIR, outpath

  ; If no logfile path is specified, put it in the output directory
  IF KEYWORD_SET(logfile) EQ 0 THEN logfile=outpath+PATH_SEP()+'log.txt'
  
  ; We'll pass a pointer to this structure around to the
  ; various functions
  distinfo = { $
    s_file:s_file, $
    appname:appname, $
    outdir:outdir, $
    outpath:outpath, $
    mk_rt_dir:!DIR + '/bin/make_rt/', $
    m_file:m_file, $
    idldir:idldir, $
    logfile:logfile, $
    rtMode:rtMode, $
    win64:KEYWORD_SET(win64), $
    macint64:KEYWORD_SET(macint64), $
    macarm64:KEYWORD_SET(macarm64), $
    lin64:KEYWORD_SET(lin64), $
    unix:0, $
    mac:0, $
    dicomex:dicomex, $
    dataminer:dataminer, $
    idl_help:0, $
    hires_maps:hires_maps, $
    hires_shapefiles:hires_shapefiles, $
    python: keyword_set(python), $
    video:video, $
    grib:grib, $
    browser:keyword_set(browser) $
  }

  p_distinfo = PTR_NEW(distinfo)
  
  check_current_platform, p_distinfo
  
  IF obsolete_message THEN BEGIN
    md_log, obsolete_message, distinfo.logfile
  ENDIF
  
  md_log, 'Starting: '+SYSTIME(/utc), distinfo.logfile
  md_log, 'Using Save file: '+distinfo.s_file, distinfo.logfile
  md_log, 'Using manifest file: '+distinfo.m_file, distinfo.logfile
  md_log, 'Using IDL directory: '+distinfo.idldir, distinfo.logfile

  md_log, copy_manifest(p_distinfo, VERBOSE = verbose), distinfo.logfile
  md_log, copy_licenseutils(p_distinfo), distinfo.logfile
  IF (distinfo.s_file NE '') THEN BEGIN
    md_log, copy_savefile(p_distinfo), distinfo.logfile
  ENDIF
  md_log, copy_launcher(p_distinfo), distinfo.logfile
  
  md_log, 'Finished: '+SYSTIME(/utc), distinfo.logfile
  md_log, ['--------------------------------------', ' '], distinfo.logfile
  
  PRINT, 'make_dist routine finished. See log file: ', distinfo.logfile
END
