; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;+
; NAME:
;   iContour
;
; PURPOSE:
;   Implements the icontour wrapper interface for the tools sytem.
;
; CALLING SEQUENCE:
;   IContour
;
; INPUTS:
;   Z[,X,Y] [,...] (see IDLgrContour)
;
; KEYWORD PARAMETERS:
;   IDENTIFIER  [out] - The identifier of the created tool.
;
;   RGB_TABLE
;   Set this keyword to the number of the predefined IDL color
;   table (0 to 40), or to either a 3 by 256 or 256 by 3 array containing
;   color values to use for contour level colors.
;
;   RGB_INDICES
;   Set this keyword to a vector of indices into the color table
;   to select colors to use for vertex colors.  If the number of
;   colors selected is less than the number of vertices, the
;   colors are repeated cyclically.
;
;   All other keywords are passed to the tool during creation.
;
; MODIFICATION HISTORY:
;   Written by:  Alan, RSI, January 2003
;   Modified: CT, Oct 2006: Added helper function, TEST keyword,
;       allow RGB_TABLE to be a Loadct table number.
;
;-


;-----------------------------------------------------------------------
; Helper routine to construct the parameter set.
; If no parameters are supplied then oParmSet will be undefined.
;
function iContour_GetParmSet, oParmSet, z, x, y, $
    TEST=test, $
    RGB_TABLE=rgbTableIn, $
    RGB_INDICES=rgbIndices, $
    NODATA=noDataIn, XRANGE=xr, YRANGE=yr, ZRANGE=zr, $
    OVERPLOT=overplot, FILL=fill

    compile_opt idl2, hidden

    if (Keyword_Set(test)) then begin
      z = 1000*MIN_CURVE_SURF(RANDOMU(1,9,9), NX=100, NY=100)
    endif
    
    if N_ELEMENTS(rgbTableIn) eq 0 then begin
      rgbTableIn = KEYWORD_SET(fill) ? COLORTABLE(49, /REVERSE) : COLORTABLE(55)
    endif

    if (N_Elements(z) eq 0) then return, 0b

    noData = KEYWORD_SET(noDataIn)
    hasXrange = ISA(xr)
    hasYrange = ISA(yr)
    hasZrange = ISA(zr)

    unknownData = 0b

    oParmSet = OBJ_NEW('IDLitParameterSet', $
        NAME='Contour parameters', $
        ICON='bullseye.svg', $
        DESCRIPTION='Contour parameters')
    oParmSet->SetAutoDeleteMode, 1

    case SIZE(z, /N_DIMENSIONS) of

    1: begin
        nx = N_ELEMENTS(x)
        ny = N_ELEMENTS(y)
        nz = N_ELEMENTS(z)
        if ((nx eq ny) && (ny eq nz)) then BEGIN
            oDataX = OBJ_NEW('IDLitDataIDLVector', $
                                 NAME='VERT X', $
                                 REFORM(x, nz))
            oDataY = OBJ_NEW('IDLitDataIDLVector', $
                                 NAME='VERT Y', $
                                 REFORM(y, nz))
            oDataZ = OBJ_NEW('IDLitDataIDLVector', $
                                 NAME='VERT Z', $
                                 REFORM(z, nz))
            ; Do not give parameter names when adding,
            ; since these need to be gridded, and are not
            ; valid contour parameters.
            oParmSet->Add, oDataX
            oParmSet->Add, oDataY
            oParmSet->Add, oDataZ
            ; Fire up the unknown data wizard after starting the tool.
            unknownData = 1
        endif else begin
            MESSAGE, 'Arguments have invalid dimensions'
        endelse

        ; auto range for /NODATA
        if (noData) then begin
          if (~keyword_set(overplot)) then begin
            if (~hasXrange) then xr = [x[0], x[-1]]
            if (~hasYrange) then yr = [y[0], y[-1]]
            if (~hasZrange) then zr = [MIN(z, MAX=mx), mx]
          endif
        endif
    end

    2: begin
        oDataZ = OBJ_NEW('IDLitDataIDLArray2d', Z, $
                            NAME='Z')
        oParmSet->add, oDataZ, PARAMETER_NAME="Z"
        zDims = size(z, /dimensions)
        
        ; auto range for /NODATA
        if (noData) then begin
          if (~keyword_set(overplot)) then begin
            if (~hasXrange) then xr = [0, zDims[0]]
            if (~hasYrange) then yr = [0, zDims[1]]
            if (~hasZrange) then zr = [MIN(z, MAX=mx), mx]
          endif
        endif

        nx = N_ELEMENTS(X)
        ny = N_ELEMENTS(Y)
        IF (nx NE 0 || ny NE 0) THEN BEGIN
            validXY = 0b
            xDims = size(x, /dimensions)
            yDims = size(y, /dimensions)

            ; if X and Y cover the x and y dimensions, resp. of Z
            ; add them to the data set
            if ((nx eq zDims[0]) && (ny eq zDims[1])) then BEGIN
                oDataX = obj_new('idlitDataIDLVector', X, NAME='X')
                oDataY = obj_new('idlitDataIDLVector', Y, NAME='Y')
                oParmSet->add, oDataX, PARAMETER_NAME="X"
                oParmSet->add, oDataY, PARAMETER_NAME="Y"
                validXY = 1b
            endif

            ; if both X and Y exist as 2D arrays of the same dim
            ; as z add them to the data set
            IF array_equal(zDims,xDims) && array_equal(zDims,yDims) THEN BEGIN
              oDataX = obj_new('IDLitDataIDLArray2D',  X, NAME='X')
              oDataY = obj_new('IDLitDataIDLArray2D', Y, NAME='Y')
              oParmSet->add, oDataX, PARAMETER_NAME="X"
              oParmSet->add, oDataY, PARAMETER_NAME="Y"
              validXY = 1b
            ENDIF

            IF ~validXY THEN BEGIN
              MESSAGE, 'X or Y argument has invalid dimensions'
            ENDIF

            ; auto range for /NODATA
            if (noData) then begin
              if (~keyword_set(overplot)) then begin
                if (~hasXrange) then xr = [MIN(x, MAX=mx), mx]
                if (~hasYrange) then yr = [MIN(y, MAX=mx), mx]
                ; We already did Z above.
              endif
            endif
        endif
    end

    else: MESSAGE, 'First argument has invalid dimensions'

    ENDCASE

    ; Check for color table. If set, add that to the data container.
    if (N_Elements(rgbTableIn) gt 0) then begin
        rgbTable = rgbTableIn
        if (N_Elements(rgbTable) eq 1) then $
            rgbTable = Colortable(rgbTable[0])
        if (SIZE(rgbTable, /N_DIMENSIONS) EQ 2) then begin
            dim = SIZE(rgbTable, /DIMENSIONS)
            ;; Handle either 3xM or Mx3, but convert to 3xM to store.
            is3xM = dim[0] eq 3
            if ((is3xM || (dim[1] eq 3)) && (MAX(dim) le 256)) then begin
                tableEntries = is3xM ? rgbTable : TRANSPOSE(rgbTable)
            endif
        endif
        if (N_Elements(tableEntries) gt 0) then begin
            ramp = BINDGEN(256)
            palette = TRANSPOSE([[ramp],[ramp],[ramp]])
            palette[*,0:N_Elements(tableEntries[0,*]) -1] = tableEntries
            oPalette = OBJ_NEW('IDLitDataIDLPalette', $
                palette, NAME='Palette')
            oParmSet->Add, oPalette, PARAMETER_NAME="PALETTE"
        endif else begin
            MESSAGE, "Incorrect dimensions for RGB_TABLE."
        endelse
    endif

    ; Check for color table indices. If set, add that to the data container.
    if (SIZE(rgbIndices,/n_dimensions))[0] EQ 1 then begin
      oColorIndices = OBJ_NEW('IDLitDataIDLVector', rgbIndices, $
                      NAME='RGB Indices', TYPE='IDLVECTOR', icon='palette.svg')
      oParmSet->add, oColorIndices, PARAMETER_NAME="RGB_INDICES"
    endif

    return, unknownData

end


;-------------------------------------------------------------------------
pro icontour, z, x, y, $
    C_VALUE=cValue, $
    N_LEVEL=nLevel, $
    DEBUG=debug, $
    IDENTIFIER=IDENTIFIER, $
    NODATA=noData, $
    OVERPLOT=overplot, $
    RGB_TABLE=rgbTableIn, $
    XSTYLE=xstyle, YSTYLE=ystyle, ZSTYLE=zstyle, $
    _REF_EXTRA=_extra

    compile_opt hidden, idl2

; Note: The error handler will clean up the oParmSet container.
@idlit_itoolerror.pro

    unknownData = iContour_GetParmSet(oParmSet, z, x, y, $
      NODATA=noData, XRANGE=xr, YRANGE=yr, ZRANGE=zr, $
      RGB_TABLE=rgbTableIn, OVERPLOT=overplot, _EXTRA=_extra)

    ; The default for Contour is X/Y/ZSTYLE=0 ("nice" range)
    if (~ISA(overplot) || ~KEYWORD_SET(overplot)) then begin
      if (~ISA(xstyle)) then xstyle = 0
      if (~ISA(ystyle)) then ystyle = 0
      if (~ISA(zstyle)) then zstyle = 0
    endif

    ; For /NODATA we do not want to set [XYZ]RANGE because that will turn off
    ; automatic range updates. Instead, use the *_MINIMUM/*_MAXIMUM keywords.
    if (ISA(xr)) then begin
      xmin = xr[0]
      xmax = xr[1]
    endif
    if (ISA(yr)) then begin
      ymin = yr[0]
      ymax = yr[1]
    endif
    if (ISA(zr)) then begin
      zmin = zr[0]
      zmax = zr[1]
    endif

    if (~ISA(cValue) && ~ISA(nLevel) && ~KEYWORD_SET(noData)) then begin
      nLevel = 0
    endif

    ; Send the data to the system for tool creation
    IDENTIFIER = IDLitSys_CreateTool("Contour Tool", $
        VISUALIZATION_TYPE="CONTOUR", $
        UNKNOWN_DATA=unknownData, $
        INITIAL_DATA=oParmSet, $
        C_LABEL_SHOW=[1,0], $
        C_VALUE=cValue, $
        N_LEVELS=nLevel, $
        NODATA=noData, $
        X_MINIMUM=xMin, X_MAXIMUM=xMax, $
        Y_MINIMUM=yMin, Y_MAXIMUM=yMax, $
        Z_MINIMUM=zMin, Z_MAXIMUM=zMax, $
        OVERPLOT=overplot, XSTYLE=xstyle, YSTYLE=ystyle, ZSTYLE=zstyle, $
        WINDOW_TITLE='IDL iContour',_EXTRA=_extra)

end
