; Copyright (c)  NV5 Geospatial Solutions, Inc. All
; rights reserved. Unauthorized reproduction is prohibited.
;
;+
; CLASS_NAME:
;    ASDF_NDArray
;
; PURPOSE:
;    The ASDF_NDArray class is used to create an ASDF ndarray object,
;    which is a subclass of YAML_Map.
;
; CATEGORY:
;    Datatypes
;
;-


; ---------------------------------------------------------------------------
function ASDF_NDArray::Init, data, _EXTRA = ex
  compile_opt idl2, hidden
  on_error, 2
  !null = self.Dictionary::Init()
  self.compression = 'none'  
  self.ASDF_NDArray::SetProperty, data = data, tag='!core/ndarray-1.1.0', _EXTRA = ex
  if (~isa(data)) then self.SetData
  return, 1
end


;TODO does not need to be looking at the actual data. could settle for metadata
; ---------------------------------------------------------------------------
function ASDF_NDArray::_RetrieveDataLength, data
  compile_opt idl2, hidden
  
  nbytesPerElement = ([0,1,2,4,4,8,8,1,0,16,0,0,2,4,8,8])[data.type]
  
  if isa(data, /string) then begin
    nbytesPerElement = max(strlen(data))
  endif
  
  n = data.length
  dataLength = n * nbytesPerElement
  return, dataLength
end


; ---------------------------------------------------------------------------
; Makes a template for openu to read into.
; FLAG_ENDIAN: returns 0,1, or -1. 0 for the data is in little endian. 1 for big
; and -1 for mixed.
; DATAISFROMIDL: This is a input flag to signal the data is coming from IDL not
; from a file
; FLAG_STRING: This comes out as 0 if a string is not in the data. and a 1
; if a string is found.
;
function ASDF_NDArray::_MakeStructTemplate, obj, FLAG_ENDIAN = flag_endian, $
  DATAISFROMIDL = fromidl, FLAG_STRING = stringfound
  compile_opt idl2, hidden
  
  if obj.hasKey('byteorder') then begin
    big_endian = (strlowcase(obj["byteorder"]) eq "big") ? 1 : 0
  endif else begin
    big_endian = 0
  endelse
  
  if isa(flag_endian) then begin
    if flag_endian ne -1 && flag_endian ne big_endian then begin
        flag_endian = -1
    endif
  endif else begin
    flag_endian = big_endian
  endelse
  
  if ~obj.hasKey('datatype') then begin
    message, `DATATYPE key is missing from asdf_ndarray.`
  endif
  
  struct = {}
  stringfound = 0
  foreach element, obj["datatype"] do begin
    ; make the template.
    ; ----
    
    ;check for nested structure.
    if isa(element["datatype"], 'LIST') && ((element["datatype"])[0] ne "ascii") then begin
      Childstruct = self._MakeStructTemplate(element, $
        FLAG_ENDIAN = flag_endian, DATAISFROMIDL = fromidl, $
        FLAG_STRING = stringfound)
      
      struct = create_struct(struct, element["name"], Childstruct, /Preserve_case)
    endif else begin
      
      if element.hasKey('byteorder') then begin
        big_endian = (strlowcase(element["byteorder"]) eq "big")

        if flag_endian ne -1 && flag_endian ne big_endian then begin
          flag_endian = -1
        endif
      endif
      
      idltype = self._DatatypeToIDLTypeCode(element["datatype"], NUMBYTES = nbytesPerElement)
      
      ; Here we are filling the template with the types the bytes should be.
      ; this, combined with readu will blitz the bytes into the proper spot with the
      ; proper IDL interpretation.

      if idltype eq 7 then begin
        ; mark that we found a string. This flag signals that the struct has
        ; a string represented as bytes. Thus we have to copy over.
        
        ; we are using byte arrays instead of strings to null pad the end of
        ; our strings
        stringfound = 1
        
        if keyword_set(fromidl) then begin
          buff_data = (nbytesPerElement eq 0) ? "" : bytarr(nbytesPerElement)
        endif else begin
          buff_data = (nbytesPerElement eq 0) ? "" : (" ").dup(nbytesPerElement)
        endelse
      endif else begin
        buff_data = fix(0, type = idltype)
      endelse

      ; TODO support shape and bool8?

      struct = create_struct(struct, element["name"], buff_data, /Preserve_case)
    endelse
  endforeach
  return, struct
end


; ---------------------------------------------------------------------------
function ASDF_NDArray::_RecursiveCopyStruct, data, newdata
  compile_opt idl2, hidden
  
  structNames = tag_names(newdata)
  for i = 0, n_elements(structNames) - 1 do begin
    
    if (isa(data[0].(i), "STRUCT")) then begin
      newdata.(i) = self._RecursiveCopyStruct(data.(i), newdata.(i))
    endif else if isa(data[0].(i), /string) && (isa(newdata[0].(i), "BYTE")) then begin
      ; we dont want to convert to bytes if tag data is entirely comprised
      ; of empty strings. This is because byte("") = 0 not nothing.
      newdata.(i) = byte(data.(i))
    endif else begin 
      newdata.(i) = data.(i)
    endelse
    
  endfor
  
  return, newdata
end


; ---------------------------------------------------------------------------
; Outline
; Create Struct copy.
; The padding will short circuit if there are no strings.
; The copy should have a byte array for strings in length equal to the max of the string.
pro ASDF_NDArray::_PadStructureStrings, data
  compile_opt idl2, hidden
  
  StructTemplate = self._MakeStructTemplate(self, $
    /DATAISFROMIDL, FLAG_STRING = stringfound)
  
  ; if no strings then we dont have to pad anything
  if ~stringfound then return
  
  ; TODO just because we have a string doesn't mean we want to copy it.
  ; we only want to copy if the strlen is less then the max. otherwise short circuit.
  
  data = self._RecursiveCopyStruct(data, replicate(StructTemplate, data.length))
  
end


; ---------------------------------------------------------------------------
pro ASDF_NDArray::_WriteData, filename
  compile_opt idl2, hidden
  on_error, 2
  
  data = self.GetData()
  
  obj = self
  if isa(data, 'STRUCT') then begin
    self._PadStructureStrings, data
  endif else if (isa(data, /STRING)) then begin
    ; This will convert our string vector into a two-dimensional array
    ; where the first dimension is the size of the longest string.
    ; Strings that are shorter than this will be padded with null chars.
    data = byte(data)
  endif
  
  ; Assemble and write block.
  ; if the write fails force a close.
  catch, err
  if (err ne 0) then begin
    catch, /cancel
    free_lun, lun, /force
    message, /reissue_last
  endif else begin

    header = self._GetBlockHeader(self._datasize)
    
    if (self.compression eq "zlib" || self.compression eq "bzp2") then begin
      
      case self.compression of
        "zlib":  data = zlib_compress(data)
        "bzp2": data = bzip2_compress(data)
        else: ; assume no compression
      endcase
      
      ; for reading
      ; support reading streamed data arrays where the dimensions are not fixed
      header.used_size = swap_endian(ulong64(n_elements(data)), /swap_if_little_endian)
    endif

    ; write byte headers first as they are not to be compressed
    openw, lun, filename, /get_lun, /append
    writeu, lun, header
    free_lun, lun, /force
    
    openw, lun, filename, /get_lun, /append
    writeu, lun, data
    free_lun, lun, /force
  endelse  
  catch, /cancel

end


; ---------------------------------------------------------------------------
; Take in a ASDF_NDArray and return a block header
;
function ASDF_NDArray::_GetBlockHeader, dataLength
  compile_opt idl2, hidden
  
  data_size = swap_endian(ulong64(dataLength), /swap_if_little_endian)
  used_size = swap_endian(ulong64(dataLength), /swap_if_little_endian)
  
  ; set up compression keyword
  compression = self.compression
  case compression of
    '': compression = bytarr(4)
    'none': compression = bytarr(4)
    'bzp2': compression = byte('bzp2')
    'zlib':  compression = byte('zlib')
    else: begin
      message, `Unsupported compression '${compression}'. Defaulting to zlib.`, /informational
      compression = byte('zlib')
    end
  endcase
  
  blockHeader = { $
    magic: byte([0xd3, 0x42, 0x4c, 0x4b]), $
    header_size: swap_endian(48us, /swap_if_little_endian), $
    flags: swap_endian(0ul, /swap_if_little_endian), $
    compression: compression, $
    alloc_size: used_size, $
    used_size: used_size, $
    data_size: data_size, $
    checksum: bytarr(16) $   ; TODO: calculate checksum
  }
  return, blockHeader
end


; ---------------------------------------------------------------------------
function ASDF_NDArray::_DatatypeToIDLTypeCode, datatype,$
  NUMBYTES = nbytesPerElement
  compile_opt idl2, hidden
  
  ; Map ASDF datatypes to IDL types - index gives the IDL type number
  asdfToIDL = ['', 'uint8', 'int16', 'int32', 'float32', 'float64', $
    'complex64', '', '', 'complex128', '', '', 'uint16', 'uint32', $
    'int64', 'uint64', '']
  idltype = (where(asdfToIDL eq datatype))[0]

  ;data type not of expected type. Grab type and handle.
  if (idltype le 0) then begin
    case datatype[0] of
      'int8': idltype = 1
      'bool8': idltype = 1
      "ascii": begin
        ; the first element for ascii will be "ascii". The second element
        ; gives the num of characters.
        idltype = 7
        nbytesPerElement = datatype[1]
      end
      else: message, `Unsupported datatype: '${datatype}'`
    endcase
  endif
  
  nbytesPerElement = isa(nbytesPerElement) ? $
    nbytesPerElement : ([0,1,2,4,4,8,8,1,0,16,0,0,2,4,8,8])[idltype]
  
  return, idltype
end


; ---------------------------------------------------------------------------
; take in byte data for one unit of type.
function ASDF_NDArray::ConvertToIDL, obj, data
  compile_opt idl2, hidden
  on_error, 2
  
  datatype = obj["datatype"]
  idltype = self._DatatypeToIDLTypeCode(datatype, NUMBYTES = nbytesPerElement)
  
  n = data.LENGTH
  data = fix(temporary(data), 0, n / nbytesPerElement, type = idltype) 

  if (obj.hasKey('strides')) then begin
    strides = obj["strides"] / nbytesPerElement
    offset = obj.hasKey('offset') ? (obj["offset"] / nbytesPerElement) : 0
    for d = 0, n_elements(offset) - 1 do begin
      case d of
        0: data = data[offset[d] : * : strides[d], *, *, *, *, *, *, *]
        1: data = data[*, offset[d] : * : strides[d], *, *, *, *, *, *]
        2: data = data[*, *, offset[d] : * : strides[d], *, *, *, *, *]
        3: data = data[*, *, *, offset[d] : * : strides[d], *, *, *, *]
        4: data = data[*, *, *, *, offset[d] : * : strides[d], *, *, *]
        5: data = data[*, *, *, *, *, offset[d] : * : strides[d], *, *]
        6: data = data[*, *, *, *, *, *, offset[d] : * : strides[d], *]
        7: data = data[*, *, *, *, *, *, *, offset[d] : * : strides[d]]
        else: message, `Unsupported number of dimensions: ${d}`
      endcase
    endfor
  endif

  if (obj.hasKey('byteorder')) then begin
    case obj["byteorder"] of
      'little': swap_endian_inplace, data, /swap_if_big_endian
      'big': swap_endian_inplace, data, /swap_if_little_endian
      else:
    endcase
    ; Now reset our flag to match our platform.
    isLittle = swap_endian(1L, /swap_if_little_endian) ne 1L
    obj["byteorder"] = isLittle ? 'little' : 'big'
  endif

  if (obj.hasKey('shape')) then begin
    shape = obj.shape
    ; If we have data from a "stream" file we will need to determine
    ; the actual size of the first dimension.
    if (isa(shape[0], /string) && shape[0] eq '*') then begin
      ; Just get rid of the first value so we can use ::toArray
      shape[0] = 0
      shape = shape.toArray()
      shape[0] = n_elements(data) / product(shape[1: *])
      ; Replace our shape property with our actual shape.
      obj.shape = shape
    endif
    data = reform(data, reverse(shape), /overwrite)
  endif

  if (datatype eq 'bool8') then begin
    data = boolean(temporary(data))
  endif
  
  return, data
end


; ---------------------------------------------------------------------------
; Internal helper routine to convert from an ASDF numeric array to an
; IDL array.
function ASDF_NDArray::_GetArrayDataFromNumeric
  compile_opt idl2, hidden
  
  ; If we have compression, we cannot use the openr, /compress flag because
  ; it doesn't work with point_lun. Instead, just read in the raw zlib
  ; compressed bytes and then uncompress them.

  if (self._datasize eq 0) then begin
    self._datasize = (file_info(self._filename)).size - self._datastart
    if (self._datasize le 0) then begin
      message, `Unable to read data block from '${self._filename}'`
    endif
  endif

  data = bytarr(self._datasize, /nozero)
  openr, lun, self._filename, /get_lun
  point_lun, lun, self._datastart
  readu, lun, data
  free_lun, lun, /force

  case self.compression of
    "zlib":  data = zlib_uncompress(data)
    "bzp2": data = bzip2_uncompress(data)
    else: ; assume no compression
  endcase
  
  data = self.ConvertToIDL(self, data)  
  return, data
end


; ---------------------------------------------------------------------------
; Internal helper routine to convert from an ASDF ascii array to an
; IDL string array.
function ASDF_NDArray::_GetStringDataFromAscii
  compile_opt idl2, hidden
  
  obj = self
  datatype = obj.datatype

  ; See if we have an ASDF ascii datatype of the form ["ascii",nn]
  ; where nn is the length of each string.
  if (n_elements(datatype) ne 2 || datatype[0] ne "ascii") then begin
    catch, /cancel
    message, `Unsupported datatype: '${datatype}'`
  endif

  data = bytarr(self._datasize, /nozero)
  openr, lun, self._filename, /get_lun
  point_lun, lun, self._datastart
  readu, lun, data
  free_lun, lun, /force
  
  case self.compression of
    "zlib":  data = zlib_uncompress(data)
    "bzp2": data = bzip2_uncompress(data)
    else: ; assume no compression
  endcase

  asciiLength = datatype[1]
  data = string(reform(data, asciiLength, n_elements(data) / asciiLength))
  return, data
end


; ---------------------------------------------------------------------------
; This will read through an array of structures and swap the endian of data
; inside it to match the system. We do not do this by default due to speed and
; size considerations.
; We modify by reference to avoid copying where possible for size consideration.
;
pro ASDF_NDArray::_HandleMixedEndian, data, obj
  compile_opt idl2, hidden
  
  ; Each struct should have a byte order on the parent. It lies! If the data in
  ; the structure doesn't have any byte order we will use it as a backup.
  if obj.hasKey("byteorder") then begin
    DefaultEndian = (strlowcase(obj["byteorder"]) eq "big") ? 1 : 0
  endif else begin
    ; In the case all else fails, assume little endian.
    DefaultEndian = 0
  endelse

  for i = 0, n_elements(tag_names(data[0])) - 1 do begin
    objDataType = (obj["datatype"])[i]
    
    if (isa(data.(i), "STRUCT")) then begin
      ; Passing data.(i) directly erases it on return as it's a temporary variable.
      ; Instead pass it in as a named variable.
      temp = data.(i)
      self._HandleMixedEndian, temp, objDataType
      data.(i) = temporary(temp)
    endif else begin
      
      if objDataType.hasKey("byteorder") then begin
        dataIsBigEndian = (strlowcase(objDataType["byteorder"]) eq "big") ? 1 : 0
      endif else begin
        dataIsBigEndian = DefaultEndian
      endelse

      systemIsBigEndian = swap_endian(1L, /swap_if_big_endian) ne 1L

      if (systemIsBigEndian ne dataIsBigEndian) then begin
        data.(i) = swap_endian(data.(i))
      endif
      
    endelse
    
  endfor

end


; ---------------------------------------------------------------------------
; Internal helper routine to convert from an ASDF numpy dtype array to an
; IDL structure array.
function ASDF_NDArray::_GetStructDataFromNumpyDType
  compile_opt idl2, hidden

  obj = self
  struct = self._MakeStructTemplate(obj, FLAG_ENDIAN = flag_endian)
  
  ; check flag_endian for mixed endian.
  if flag_endian eq -1 then begin
    swap_endian = 0
  endif else begin
    systemIsBigEndian = swap_endian(1L, /swap_if_big_endian) ne 1L
    swap_endian = systemIsBigEndian ne flag_endian
  endelse
  
  data = replicate(struct, obj["shape"])

  if (self.compression eq "zlib" || self.compression eq "bzp2") then begin

    ; To handle complex numpy datatypes which will turn into IDL structures,
    ; and that are also compressed, we need to read the compressed bytes,
    ; run them through zlib_uncompress, write them out to a temp file
    ; and then we can finally read them directly into the IDL struct array.
    ; We cannot use openr, /compress flag because it doesn't work
    ; with point_lun.
    tmpdata = bytarr(self._datasize, /nozero)
    openr, lun, self._filename, /get_lun
    point_lun, lun, self._datastart
    readu, lun, tmpdata
    free_lun, lun, /force
    
    case self.compression of 
      "zlib":  tmpdata = zlib_uncompress(tmpdata)
      "bzp2": tmpdata = bzip2_uncompress(tmpdata)
      else: ; assume no compression
    endcase

    tmpfile = filepath(`idltmpstruct_${systime(1)}.dat`, /tmp)
    openw, lun, tmpfile, /get_lun
    writeu, lun, tmpdata
    free_lun, lun, /force

    openr, lun, tmpfile, /get_lun, swap_endian = swap_endian
    readu, lun, data
    free_lun, lun, /force
    file_delete, tmpfile, /quiet

  endif else begin
    openr, lun, self._filename, /get_lun, swap_endian = swap_endian
    point_lun, lun, self._datastart
    readu, lun, data
    free_lun, lun, /force
  endelse
  
  ;clean up mixed endian
  if flag_endian eq -1 then begin
    self._HandleMixedEndian, data, obj
  endif
  
  return, data
end


; ---------------------------------------------------------------------------
function ASDF_NDArray::GetData
  compile_opt idl2, hidden
  on_error, 2

  ; Is the data embedded within the ASDF file?
  if (self.hasKey('data')) then begin
    self.Dictionary::GetProperty, data = data
    return, data
  endif
  
  obj = self
  
  ; we will take care of actually (un)compressing during the read in.
  supportedCompressions = ["", "none", "zlib", "bzp2"]
  if (~supportedCompressions.hasvalue(self.compression)) then begin
    catch, /cancel
    message, `Compression not supported: '${self.compression}'`
  endif
  
  datatype = obj.datatype

  ; See if we have a complex numpy datatype, which will become an IDL structure.
  ; These are technically YAML_SEQUENCE and YAML_MAP, but those are
  ; subclasses of LIST and HASH, so use the more generic classes.
  if (isa(datatype, 'LIST') && isa(datatype[0], 'HASH')) then begin
    data = self._GetStructDataFromNumpyDType()
  endif else if (isa(datatype, 'LIST')) then begin
    data = self._GetStringDataFromAscii()
  endif else begin
    data = self._GetArrayDataFromNumeric()
  endelse
  
  ; Set using brackets so our key is lowercase
  obj['data'] = data
  return, data
end


; ---------------------------------------------------------------------------
; an array of struct is how we are handling ndarrays in IDL.
function ASDF_NDArray::_CreateDatatypeForStruct, data
  compile_opt idl2, hidden

  datamap = ['', 'uint8', 'int16', 'int32', 'float32', 'float64', 'complex64', $
  'ascii', 'struct', 'complex128', '', '', 'uint16', 'uint32', 'int64', 'uint64']
  
  tag_names = tag_names(data[0])
  parentDataType = list()
  firstStruct = data[0]
  
  ; Make Ohash to pass to YAML
  ; -----
  for i = 0, n_elements(tag_names) - 1 do begin
    tagValue = firstStruct.(i)
    
    child = !NULL
    if isa(tagValue, "STRUCT") then begin
      child = self._CreateDatatypeForStruct(tagValue)
    endif
    
    keys = list()
    values = list()
    
    name = tag_names[i]
    
    isLittle = swap_endian(1L, /swap_if_little_endian) ne 1L
    byteOrder = isLittle ? 'little' : 'big'
    keys.add, "byteorder"
    values.add, byteOrder
    
    datatype = datamap[tagValue.type]
    if datatype eq '' then message, "Illegal datatype for structure field " + name
    
    keys.add, "datatype"
    if isa(child) then begin
      values.add, child
    endif else if datatype eq "ascii" then begin
      ; ascii is the only ASDF type to be written out in the format of list.
      ; EX: ["ascii",23]
      ; we store the max here because we need the structs to be all the same
      ; size for reading and writing.
      tempDataLength = self._RetrieveDataLength(data.(i))
      ; tmpDatalength already includes data.length so we want to back it out.
      values.add, list(datatype, tempDataLength / data.length)
      self._datasize += tempDataLength
    endif else begin
      values.add, datatype
      self._datasize += self._RetrieveDataLength(tagValue) * data.length
    endelse
    
    keys.add, "name"
    values.add, name
    
    parentSubDataType = orderedhash(keys, values)
    parentDataType.add, parentSubDataType
  endfor
  
  return, parentDataType
end


; ---------------------------------------------------------------------------
pro ASDF_NDArray::SetData, data
  compile_opt idl2, hidden
  on_error, 2

  obj = self
  ; Set all of these using brackets so our keys are lowercase
  if (isa(data)) then begin
    obj['shape'] = isa(data,/scalar) ? [1] : reverse(size(data, /dimensions))
    ; Reset our flag to match our platform.
    isLittle = swap_endian(1L, /swap_if_little_endian) ne 1L
    obj['byteorder'] = isLittle ? 'little' : 'big'
    datamap = ['', 'uint8', 'int16', 'int32', 'float32', 'float64', 'complex64', $
      '', '', 'complex128', '', '', 'uint16', 'uint32', 'int64', 'uint64']

    ; Reset in case we are setting new data on an existing ASDF_NDArray
    self._datasize = 0
    if isa(data,"STRUCT") then begin
      obj["datatype"] = self._CreateDatatypeForStruct(data)
    endif else if isa(data, /string) then begin
      len = max(strlen(data))
      obj['datatype'] = list('ascii', len)
      self._datasize = len * n_elements(data)
    endif else begin
      obj['datatype'] = datamap[data.type]
      self._datasize = self._RetrieveDataLength(data)
    endelse

    obj['data'] = isa(data, /scalar) ? [data] : data
    
    ; removing source at any point will result in the data being "inline"
    
    if (~self._source_set) then begin
      if (data.length gt 10 || isa(data, "STRUCT")) then begin
        obj["source"] = 0
      endif
    endif
  
  endif else begin
    obj['shape'] = [0]
    obj['byteorder'] = 'little'
    obj['datatype'] = 'float64'
  endelse
end


; ---------------------------------------------------------------------------
function ASDF_NDArray::_overloadForeach, value, key
  compile_opt idl2, hidden
  on_error, 2
  
  result = self.Dictionary::_overloadForeach(value, key)
  ; If we have a 'source' then skip the data field
  if (key.toLower() eq 'data') then begin
    if (self.hasKey('source')) then begin
      result = self.Dictionary::_overloadForeach(value, key)
    endif else begin
      ; Normally, yaml_serialize would write out a byte array as !!binary.
      ; Instead, return an integer array to fool it into writing out numbers.
      ; This should not affect users getting the data via ['data'] or .data
      ; because that will go through ::GetData up above and return a bytarr.
      if (isa(value, 'byte') && ~isa(value, /boolean)) then begin
        value = fix(value)
      endif
    endelse
  endif
  return, result
end


; ---------------------------------------------------------------------------
function ASDF_NDArray::Get, key
  compile_opt idl2, hidden
  on_error, 2

  if (key.toLower() eq 'data') then begin
    return, self->GetData()
  endif
  
  result = self.Dictionary::Get(key)
  return, result
end


; ---------------------------------------------------------------------------
pro ASDF_NDArray::GetProperty, $
  alias=alias, anchor=anchor, tag=tag, value=value, $
  storage=storage, data=data, $
  _datastart=_datastart, _datasize=_datasize, _filename=_filename, $
  compression=compression, _ref_extra=ex
  compile_opt idl2, hidden
  on_error, 2

  ; Need to handle all of these so they don't become keys in the dictionary.
  self.YAML_Node::GetProperty, alias=alias, anchor=anchor, tag=tag, value=value
  if arg_present(_filename) then _filename = self._filename
  if arg_present(_datastart) then _datastart = self._datastart
  if arg_present(_datasize) then _datasize = self._datasize
  if arg_present(compression) then compression = self.compression
  if arg_present(data) then begin
    data = self->GetData()
  endif

  if (arg_present(storage)) then begin
    obj = self
    source = obj.hasKey('source') ? obj['source'] : !null
    storage = 'inline'
    if (isa(source, /number)) then begin
      storage = 'internal'
    endif else if (isa(source, /string)) then begin
      storage = 'external'
    endif
  endif

  if (isa(ex)) then begin
    self.Dictionary::GetProperty, _extra=ex
  endif
end


; ---------------------------------------------------------------------------
pro ASDF_NDArray::SetProperty, $
  alias=alias, anchor=anchor, tag=tag, value=value, $
  storage=storage, data=data, source=source, $
  _datastart=_datastart, _datasize=_datasize, _filename=_filename, $
  compression=compression, _extra=ex
  compile_opt idl2, hidden
  on_error, 2

  ; Need to handle all of these so they don't become keys in the dictionary.
  self.YAML_Node::SetProperty, alias=alias, anchor=anchor, tag=tag, value=value
  obj = self
  if (isa(storage)) then begin
    tmpSource = obj.hasKey('source') ? obj['source'] : !null
    case storage.toLower() of
      'inline': if ~isa(tmpSource) then obj.Remove, 'source'
      'internal': if (~isa(tmpSource, /number)) then obj['source'] = 0
      'external': if (~isa(tmpSource, /string)) then obj['source'] = ''
    endcase
    self._source_set = 1b
  endif

  if (isa(source)) then begin
    obj['source'] = source
    self._source_set = 1b
  endif

  if (isa(data)) then begin
    self->SetData, data
  endif

  if (isa(_filename)) then self._filename = _filename
  if (isa(_datastart)) then self._datastart = _datastart
  if (isa(_datasize)) then self._datasize = _datasize
  if (isa(compression)) then self.compression = compression
  if (isa(ex)) then begin
    self.Dictionary::SetProperty, _extra=ex
  endif
end


; ---------------------------------------------------------------------------
function ASDF_NDArray::_overloadHelp, varname
  compile_opt idl2, hidden
  on_error, 2
  
  result = varname + ('               ').substring(varname.strlen()) + $
  ` ${obj_class(self)}  <ID=${obj_valid(self, /get_heap_identifier)}>`
  extra = ''
  if (self.hasKey('datatype')) then extra += ` ${self['datatype']}`
  if (self.hasKey('shape')) then begin
    shape = self['shape']
    extra += shape.length gt 1 ? ` ${shape}` : ` [${shape}]`
  endif
  if (self.hasKey('source')) then begin
    extra += isa(self['source'], /number) ? ' internal' : ' external'
  endif else begin
    extra += ' inline'
  endelse
  if (extra) then result += ' ' + extra
  return, result
end


; ---------------------------------------------------------------------------
pro ASDF_NDArray__DEFINE
  compile_opt idl2, hidden
  void = {ASDF_NDArray, $
    inherits YAML_Node, $
    inherits Dictionary, $
    _filename: '', $
    _datastart: 0LL, $
    _datasize: 0LL, $
    compression: '', $
    _source_set: 0b}
end
