; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;----------------------------------------------------------------------------
; See http://www.rafekettler.com/magicmethods.html
;

;---------------------------------------------------------------------------
function Python::Init, pyObjectID

  compile_opt idl2, hidden
  ON_ERROR, 2

  PyUtils.Load

  self.pyID = -1
  if (ISA(pyObjectID) && pyObjectID ne -1) then begin
    self.pyID = pyObjectID
    ; We are caching a reference - so bump up the refcount.
    PYTHON_INCREF, self.pyID
  endif

  HEAP_NOSAVE, self
  return, 1
end


;---------------------------------------------------------------------------
pro Python::Cleanup

  compile_opt idl2, hidden
  ON_ERROR, 2
  if (self.pyID gt 0) then begin
    PYTHON_DECREF, self.pyID
  endif
end


;---------------------------------------------------------------------------
; Internal method to find a case-insensitive match of an attribute/method.
;
function Python::_FindAttr, pyID, attr, METHOD=method

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  if (ISA(self)) then begin
    MESSAGE, 'Internal error: Must be called as a static method.'
  endif

  ; Try whatever case came in (usually lowercase).
  if (PYTHON_HASATTR(pyID, attr)) then begin
    if (~KEYWORD_SET(method)) then begin
      return, attr
    endif
    pyObj = PYTHON_GETATTR(pyID, attr)
    if (PYTHON_HASATTR(pyObj.pyID, '__call__')) then begin
      return, attr
    endif
  endif

  ; If that fails, use dir() to retrieve an IDL list.
  keys = Python.Dir(Python(pyID))

  if (ISA(keys)) then begin
    attrTmp = strlowcase(attr)
    foreach key, keys do begin
      if (key.ToLower() eq attrTmp) then begin
        if (~KEYWORD_SET(method)) then begin
          return, key
        endif
        pyObj = PYTHON_GETATTR(pyID, key)
        if (PYTHON_HASATTR(pyObj.pyID, '__call__')) then begin
          return, key
        endif
      endif
    endforeach
  endif

  return, !null
end


;---------------------------------------------------------------------------
function Python::GetAttr, pyObj, attr, WRAP=wrap

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  if (ISA(self)) then begin
    MESSAGE, /noname, 'Python::GetAttr must be called as a static method: ' + $
      'result = Python.GetAttr(pyObj, "attribute")'
  endif

  if (n_params() ne 2) then begin
    MESSAGE, /noname, 'Incorrect number of arguments: ' + $
      'result = Python.GetAttr(pyObj, "attribute")'
  endif

  PyUtils.Load
  attrName = Python._FindAttr(pyObj.pyID, attr)
  return, PYTHON_GETATTR(pyObj.pyID, attrName, WRAP = wrap)
end


;---------------------------------------------------------------------------
; Note: Do not add *any* keywords here. Everything needs to be handled
; through REF_EXTRA. See the note below.
;
pro Python::GetProperty, _REF_EXTRA=extra

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  PyUtils.Load

  ; Am I being called as a static method?
  ; The "-1" indicates the builtins module
  pyID = ISA(self) ? self.pyID : -1

  ; We need to handle our own keywords as special cases of ref_extra.
  ; Otherwise, because of IDL keyword matching, if you tried to retrieve
  ; a variable named "c", then Python.C would just match the CLASSNAME keyword.
  foreach ex, extra do begin
    case ex of

      'CLASSNAME': begin
        class = PYTHON_GETATTR(pyID, '__class__')
        (SCOPE_VARFETCH('CLASSNAME', /REF_EXTRA)) = $
          PYTHON_GETATTR(class.pyID, '__name__')
      end

      'DIM': (SCOPE_VARFETCH('DIM', /REF_EXTRA)) = self._overloadSize()

      'LENGTH': begin
        dim = self._overloadSize()
        (SCOPE_VARFETCH('LENGTH', /REF_EXTRA)) = PRODUCT(dim, /INTEGER)
      end

      'NDIM': begin
        dim = self._overloadSize()
        ndim = ISA(dim, /ARRAY) ? N_ELEMENTS(dim) : 0
        (SCOPE_VARFETCH('NDIM', /REF_EXTRA)) = ndim
      end

      'PYID': (SCOPE_VARFETCH('PYID', /REF_EXTRA)) = pyID

      'REFCOUNT': begin
        sys = PYTHON_IMPORT('sys')
        refcount = PYTHON_CALLMETHOD(sys.pyID, 'getrefcount', !false, 1, self)
        ; Subtract 1 since getrefcount temporarily bumps up the ref count.
        refcount--
        (SCOPE_VARFETCH('REFCOUNT', /REF_EXTRA)) = refcount
      end

      'TNAME': (SCOPE_VARFETCH('TNAME', /REF_EXTRA)) = 'OBJREF'

      'TYPECODE': (SCOPE_VARFETCH('TYPECODE', /REF_EXTRA)) = 11L

      'TYPENAME': (SCOPE_VARFETCH('TYPENAME', /REF_EXTRA)) = OBJ_CLASS(self)

      'TYPESIZE': (SCOPE_VARFETCH('TYPESIZE', /REF_EXTRA)) = 4L

      else: begin
        exname = Python._FindAttr(pyID, STRLOWCASE(ex))
        if (~ISA(exname)) then begin
          MESSAGE, /NONAME, 'Python: Unknown attribute: "' + STRLOWCASE(ex) + '"'
        endif
        result = PYTHON_GETATTR(pyID, exname)
        (SCOPE_VARFETCH(ex, /REF_EXTRA)) = TEMPORARY(result)
      end

    endcase
  endforeach
end


;---------------------------------------------------------------------------
; Note: Do not add *any* keywords here. Everything needs to be handled
; through REF_EXTRA. See the note above in GetProperty.
;
pro Python::SetProperty, $
  _REF_EXTRA=extra

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  PyUtils.Load

  ; Am I being called as a static method?
  ; The "-1" indicates the builtins module
  pyID = ISA(self) ? self.pyID : -1

  foreach ex, extra do begin
    exname = Python._FindAttr(pyID, STRLOWCASE(ex))
    ; If we didn't find an existing attribute (of any case),
    ; create a new lowercase one.
    if (~ISA(exname)) then begin
      exname = STRLOWCASE(ex)
    endif
    PYTHON_SETATTR, pyID, exname, SCOPE_VARFETCH(ex, /REF_EXTRA)
  endforeach
end


;---------------------------------------------------------------------------
function Python::Import, module

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  if (ISA(self)) then begin
    MESSAGE, /NONAME, 'Python::Import must be called as a static method.'
  endif

  PyUtils.Load


  ; TODO: This is really bad!  We are implicity changing the package namespace from what
  ; the caller has passed in.  If the caller imports envipyengine.config, then we move
  ; the config module into the main namespace and can cause naming conflicts that would confuse
  ; the end user
  submodule = (module.Split('\.'))[-1]
  void = Python.Run('import ' + module + ' as ' + submodule)

  ; Suppress random underflows
  void = CHECK_MATH()
  result = PYTHON_GETATTR(-1, submodule)

  return, result
end


;---------------------------------------------------------------------------
pro Python::Import, module

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  if (ISA(self)) then begin
    MESSAGE, /NONAME, 'Python::Import must be called as a static method.'
  endif

  PyUtils.Load

  ; TODO: This is really really bad! Never under any circumstances use "*"!
  ; Not only are we not doing what the caller expects, whenever this method is called,
  ; we take all the functions, classes, variables in the module namespace and move them into
  ; the current namespace.  This opens the door for naming collisions.
  void = Python.Run('from ' + module + ' import *')

end


;---------------------------------------------------------------------------
pro Python::Load, version

  compile_opt idl2, hidden, static
  ON_ERROR, 2
  PyUtils.Load, version
end


;---------------------------------------------------------------------------
; Redirect Python's stdout to our own Python class, so we can capture it.
;
pro Python::_StartCall, command, CAPTURE=capture

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  lf = STRING(10b)

  ; import some common modules
  void = PYTHON_RUN('import sys')
  void = PYTHON_RUN('if sys.version_info.major == 2:' + lf + $
    '    from StringIO import StringIO' + lf + $
    'else:' + lf + $
    '    from io import StringIO')
  void = PYTHON_RUN('import idlpython')

  if (KEYWORD_SET(capture)) then begin
    ; Capture all Python output into a "data" attribute,
    ; which can be retrieved later.
    void = PYTHON_RUN( $
      "class IDL_stdout_capture():" + lf + $
      "  def __init__(self):" + lf + $
      "    self.data = ''" + lf + $
      "    self.enable = True" + lf + $
      "  def write(self, s):" + lf + $
      "    if (self.enable):" + lf + $
      "      self.data += s" + lf + $
      "  def flush(self):" + lf + $
      "    pass" + lf)
    void = PYTHON_RUN("sys.stdout_orig=sys.stdout; " + $
    "sys.stdout = IDL_stdout_capture()")
  endif else begin
    ; Catch all Python output and forward it to IDL using our idlpython
    ; module. This is in C code since it needs to call back into IDL.
    void = PYTHON_RUN( $
      "class IDL_stdout_python():" + lf + $
      "  def __init__(self):" + lf + $
      "    self.data = ''" + lf + $
      "    self.skipLF = False" + lf + $
      "  def write(self, s):" + lf + $
      "    if s != '\n':" + lf + $         ; Python sends the linefeed after
      "      idlpython.output(s)" + lf + $ ; the output line. So swallow the
      "      self.skipLF = True" + lf + $  ; next linefeed character.
      "    else:" + lf + $
      "      if not self.skipLF: idlpython.output('')" + lf + $  ; A real linefeed
      "      self.skipLF = not self.skipLF" + lf + $      ; Flip the boolean
      "  def flush(self):" + lf + $
      "    pass" + lf)
    ; This will "fail" if you are in Python->IDL->Python, but that's okay
    ; because Python will just send any Python output to itself.
    void = PYTHON_RUN( $
      "try:" + lf + $
      "  sys.stdout_orig=sys.stdout" + lf + $
      "  sys.stdout = IDL_stdout_python()" + lf + $
      "except:" + lf + $
      "  pass")
  endelse

  void = PYTHON_RUN("sys.stderr_orig = sys.stderr")
  void = PYTHON_RUN("sys.stderr = StringIO()")

end


;---------------------------------------------------------------------------
; Retrieve the Python output and restore Python's original stdout.
;
function Python::_FinishCall, command, STDERR=stderr

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  lf = STRING(10b)

  ; We might not actually have wired up our own stdout,
  ; so provide a try/except fallback.
  void = PYTHON_RUN( $
    "try:" + lf + $
    "  __idloutput__=sys.stdout.data" + lf + $
    "  sys.stdout=sys.stdout_orig" + lf + $
    "except:" + lf + $
    "  __idloutput__ = ''")
  result = PYTHON_GETATTR(-1, '__idloutput__')

  void = PYTHON_RUN("__idlstderr__ = sys.stderr.getvalue()")
  void = PYTHON_RUN("sys.stderr = sys.stderr_orig")
  stderr = PYTHON_GETATTR(-1, '__idlstderr__')

  ; Using linefeeds (\n=10b), break the output into a string array.
  lf = STRING(10b)
  result = STRTOK(result, lf, /EXTRACT, /PRESERVE_NULL)
  if (result[-1] eq '' && result.LENGTH gt 1) then begin
    result = result[0:-2]
  endif
  if (result.LENGTH eq 1) then result = result[0]
  return, result
end


;---------------------------------------------------------------------------
; Unfortunately, numpy (and other libraries) don't play nicely if
; Python is reloaded. https://bugs.python.org/issue34309
; pro Python::Reset
;   compile_opt idl2, hidden, static
;   on_error, 2
;   if (isa(self)) then begin
;     message, /NONAME, 'Python::Reset must be called as a static method.'
;   endif
;   defsysv, '!IDLPYLOADED', exists=exists
;   if ~exists then defsysv, '!IDLPYLOADED', 0
;   !IDLPYLOADED = 0
;   Python_Reset
; end


;---------------------------------------------------------------------------
function Python::Run, command, STDOUT=stdout, STDERR=stderr

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  if (ISA(self)) then begin
    MESSAGE, /NONAME, 'Python::Run must be called as a static method.'
  endif

  PyUtils.Load

  Python._StartCall, CAPTURE=~KEYWORD_SET(stdout)

  ; We need to catch errors so we can restore Python's original stdout.
  CATCH, ierr
  if (ierr ne 0) then begin
    CATCH, /CANCEL
    void = Python._FinishCall()
    MESSAGE, /REISSUE_LAST
  endif

  lf = STRING(10b)

  foreach com, command do begin
    com1 = com.Split('\\\\n')
    for i=0,com1.length-1 do begin
      com1[i] = (com1[i].Split('\\n')).Join(lf)
    endfor
    com = com1.Join('\\n')
    if (STRLEN(com.Trim()) eq 0) then continue

    ; For commands ending with a semicolon, suppress the output.
    if (com.EndsWith(';')) then begin
    void = PYTHON_RUN("sys.stdout.enable = False")
    disabledOutput = !True
  endif

  void = PYTHON_RUN(com)

  ; Reenable output?
  if (KEYWORD_SET(disabledOutput)) then begin
    void = PYTHON_RUN("sys.stdout.enable = True")
    disabledOutput = !False
  endif

endforeach

; Retrieve the Python output.
result = Python._FinishCall(STDERR=stderr)

return, result
end


;---------------------------------------------------------------------------
pro Python::Run, command

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  if (N_PARAMS() eq 1) then begin
    result = Python.Run(command)
    if (ISA(result, /ARRAY) || STRLEN(result) gt 0) then begin
      PRINT, result, /IMPLIED
    endif
    return
  endif

  cmd = ''

  catch, iErr
  if (iErr ne 0) then begin
    MESSAGE, !error_state.msg, /INFO
  endif

  while (1) do begin
    READ, cmd, PROMPT='>>> '
    if (cmd.Trim() eq '' || $
      cmd.StartsWith('quit(',/FOLD) || $
      cmd.ToLower() eq 'quit' || cmd eq '^C') then break
    if (STRLEN(cmd) eq 0) then begin
      continue
    endif
    result = Python.Run(cmd)
    if (N_ELEMENTS(result) gt 1 || STRLEN(result) gt 0) then begin
      PRINT, result, /IMPLIED
      ; IDL-69430 - work around a Windows command-line bug (69444)
      ; where output would sometimes appear after the next "read" cmd.
      WAIT, 0.01
    endif
  endwhile
end


;---------------------------------------------------------------------------
function Python::Wrap, value

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  if (ISA(self)) then begin
    MESSAGE, /NONAME, 'Python::Wrap must be called as a static method.'
  endif

  if (~ISA(value) && ~ISA(value, /NULL)) then begin
    MESSAGE, /NONAME, 'Python: Undefined variable: ' + SCOPE_VARNAME(value, LEVEL=-1)
  endif

  PyUtils.Load

  return, PYTHON_WRAP(ARG_PRESENT(value) ? value : TEMPORARY(value))
end


;----------------------------------------------------------------------------
function Python::_overloadFunction, $
  a001,a002,a003,a004,a005,a006,a007,a008,a009,a010,a011,a012,a013,a014,a015,a016, $
  a017,a018,a019,a020,a021,a022,a023,a024,a025,a026,a027,a028,a029,a030,a031,a032, $
  WRAP=wrap, _REF_EXTRA=ex

  compile_opt idl2, hidden
  ON_ERROR, 2

  PyUtils.Load

  ; Am I being called as a static method?
  ; The "-1" indicates the builtins module
  pyID = ISA(self) ? self.pyID : -1
  method = "__call__"
  nparams = N_PARAMS()

  Python._StartCall

  ; We need to catch errors so we can restore Python's original stdout.
  CATCH, ierr
  if (ierr ne 0) then begin
    CATCH, /CANCEL
    void = Python._FinishCall()
    MESSAGE, /REISSUE_LAST
  endif

  result = PYTHON_CALLMETHOD(pyID, method, keyword_set(wrap), nparams, _EXTRA=ex, $
    arg_present(a001) ? a001 : temporary(a001), $
    arg_present(a002) ? a002 : temporary(a002), $
    arg_present(a003) ? a003 : temporary(a003), $
    arg_present(a004) ? a004 : temporary(a004), $
    arg_present(a005) ? a005 : temporary(a005), $
    arg_present(a006) ? a006 : temporary(a006), $
    arg_present(a007) ? a007 : temporary(a007), $
    arg_present(a008) ? a008 : temporary(a008), $
    arg_present(a009) ? a009 : temporary(a009), $
    arg_present(a010) ? a010 : temporary(a010), $
    arg_present(a011) ? a011 : temporary(a011), $
    arg_present(a012) ? a012 : temporary(a012), $
    arg_present(a013) ? a013 : temporary(a013), $
    arg_present(a014) ? a014 : temporary(a014), $
    arg_present(a015) ? a015 : temporary(a015), $
    arg_present(a016) ? a016 : temporary(a016), $
    arg_present(a017) ? a017 : temporary(a017), $
    arg_present(a018) ? a018 : temporary(a018), $
    arg_present(a019) ? a019 : temporary(a019), $
    arg_present(a020) ? a020 : temporary(a020), $
    arg_present(a021) ? a021 : temporary(a021), $
    arg_present(a022) ? a022 : temporary(a022), $
    arg_present(a023) ? a023 : temporary(a023), $
    arg_present(a024) ? a024 : temporary(a024), $
    arg_present(a025) ? a025 : temporary(a025), $
    arg_present(a026) ? a026 : temporary(a026), $
    arg_present(a027) ? a027 : temporary(a027), $
    arg_present(a028) ? a028 : temporary(a028), $
    arg_present(a029) ? a029 : temporary(a029), $
    arg_present(a030) ? a030 : temporary(a030), $
    arg_present(a031) ? a031 : temporary(a031), $
    arg_present(a032) ? a032 : temporary(a032))

  void = Python._FinishCall()

  return, result
end


;----------------------------------------------------------------------------
function Python::_overloadMethod, methodIn, $
  a001,a002,a003,a004,a005,a006,a007,a008,a009,a010,a011,a012,a013,a014,a015,a016, $
  a017,a018,a019,a020,a021,a022,a023,a024,a025,a026,a027,a028,a029,a030,a031,a032, $
  WRAP=wrap, _REF_EXTRA=ex

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  PyUtils.Load

  ; Do not include the method name
  nparams = N_PARAMS() - 1

  ; Am I being called as a static method?
  ; The "-1" indicates the builtins module
  pyID = ISA(self) ? self.pyID : -1

  if (pyID eq 0) then begin
    MESSAGE, /NONAME, 'Python: Python object is undefined.'
  endif

  method = STRLOWCASE(methodIn)

  ; Hack for the Python builtins help() function.  Normally this would
  ; return nothing. Instead, use pydoc to construct the help output
  ; and return that as the result.
  if (~ISA(self) && method eq 'help' && nparams eq 1) then begin
    ; Use the internal _Import instead of our Import method
    ; to avoid polluting the Python namespace.
    pydoc = PYTHON_IMPORT('pydoc')
    pyID = pydoc.pyID
    method = 'render_doc'
  endif

  ; Hack for the Python builtins dir() function. With no arguments
  ; this was throwing an error about "frame does not exist".
  ; Instead, substitute __main__ for the frame to get a valid dir() result.
  if (~ISA(self) && method eq 'dir' && nparams eq 0) then begin
    a001 = Python.Import('__main__')
    nparams++
  endif

  ; Find the matching method, ignoring case.
  method = Python._FindAttr(pyID, method, /METHOD)
  if (~ISA(method)) then begin
    MESSAGE, /NONAME, 'Python: Unknown method: "' + methodIn + '"'
  endif

  Python._StartCall

  ; We need to catch errors so we can restore Python's original stdout.
  CATCH, ierr
  if (ierr ne 0) then begin
    CATCH, /CANCEL
    void = Python._FinishCall()
    MESSAGE, /REISSUE_LAST
  endif

  result = PYTHON_CALLMETHOD(pyID, method, keyword_set(wrap), nparams, _EXTRA=ex, $
    arg_present(a001) ? a001 : temporary(a001), $
    arg_present(a002) ? a002 : temporary(a002), $
    arg_present(a003) ? a003 : temporary(a003), $
    arg_present(a004) ? a004 : temporary(a004), $
    arg_present(a005) ? a005 : temporary(a005), $
    arg_present(a006) ? a006 : temporary(a006), $
    arg_present(a007) ? a007 : temporary(a007), $
    arg_present(a008) ? a008 : temporary(a008), $
    arg_present(a009) ? a009 : temporary(a009), $
    arg_present(a010) ? a010 : temporary(a010), $
    arg_present(a011) ? a011 : temporary(a011), $
    arg_present(a012) ? a012 : temporary(a012), $
    arg_present(a013) ? a013 : temporary(a013), $
    arg_present(a014) ? a014 : temporary(a014), $
    arg_present(a015) ? a015 : temporary(a015), $
    arg_present(a016) ? a016 : temporary(a016), $
    arg_present(a017) ? a017 : temporary(a017), $
    arg_present(a018) ? a018 : temporary(a018), $
    arg_present(a019) ? a019 : temporary(a019), $
    arg_present(a020) ? a020 : temporary(a020), $
    arg_present(a021) ? a021 : temporary(a021), $
    arg_present(a022) ? a022 : temporary(a022), $
    arg_present(a023) ? a023 : temporary(a023), $
    arg_present(a024) ? a024 : temporary(a024), $
    arg_present(a025) ? a025 : temporary(a025), $
    arg_present(a026) ? a026 : temporary(a026), $
    arg_present(a027) ? a027 : temporary(a027), $
    arg_present(a028) ? a028 : temporary(a028), $
    arg_present(a029) ? a029 : temporary(a029), $
    arg_present(a030) ? a030 : temporary(a030), $
    arg_present(a031) ? a031 : temporary(a031), $
    arg_present(a032) ? a032 : temporary(a032))

  void = Python._FinishCall()

  ; Hack for the builtin help() function. See comment above.
  if (method eq 'render_doc' && ISA(result, /STRING, /SCALAR)) then begin
    ; Remove all of the fancy "backspace" bold formatting codes,
    ; and split the string up at the linefeeds.
    result = BYTE(result)
    backspace = result eq 8b
    if (~ARRAY_EQUAL(backspace, 0b)) then begin
      ; Delete the previous character.
      backspace += SHIFT(backspace,-1)
      result = STRING(result[WHERE(~backspace)])
    endif
    result = STRTOK(result, STRING(10b), /EXTRACT, /PRESERVE_NULL)
  endif

  return, result
end


;---------------------------------------------------------------------------
function Python::_convertSubscript, isRange, subs, dim
  compile_opt idl2, hidden, static

  if (isRange) then begin
    ; Adjust negative indices (also handles "*" indices.
    if (subs[0] lt 0 && (ABS(subs[0]) le dim)) then subs[0] += dim
    if (subs[1] lt 0 && (ABS(subs[1]) le dim)) then subs[1] += dim
    nelem = (subs[1] - subs[0]) / subs[2] + 1
    if (nelem le 0) then begin
      MESSAGE, /NONAME, 'Python: Illegal subscript range.'
    endif
    return, Python.slice(subs[0], subs[1] + 1, subs[2])
  endif

  return, subs
end


;---------------------------------------------------------------------------
pro Python::_overloadBracketsLeftSide, arg, value, isRange, $
  i0, i1, i2, i3, i4, i5, i6, i7

  compile_opt idl2, hidden
  ON_ERROR, 2

  ; Return quietly if any subscripts are !NULL.
  if (ISA(i0, /NULL) || ISA(i1, /NULL) || ISA(i2, /NULL) || $
    ISA(i3, /NULL) || ISA(i4, /NULL) || ISA(i5, /NULL) || $
    ISA(i6, /NULL) || ISA(i7, /NULL)) then return

  self.GetProperty, CLASSNAME=pythonClass

  if (pythonClass eq 'list' || pythonClass eq 'tuple') then begin

    count = (self._overloadSize())[0]

    ; handle [i:j:k]
    if (isRange[0]) then begin
      ; Adjust negative indices (also handles "*" indices.
      if (i0[0] lt 0 && (ABS(i0[0]) le count)) then i0[0] += count
      if (i0[1] lt 0 && (ABS(i0[1]) le count)) then i0[1] += count
      nelem = (i0[1] - i0[0])/i0[2] + 1
      if (nelem le 0) then begin
        MESSAGE, /NONAME, 'Python: Illegal subscript range.'
      endif
      i0 = LINDGEN(nelem)*i0[2] + i0[0]
    endif

    ; For an array of indices, set each value.
    if (ISA(i0, /ARRAY) || ISA(i0, 'IDL_OBJECT')) then begin
      nval = N_ELEMENTS(value)
      if (nval gt 1 && nval ne N_ELEMENTS(i0)) then begin
        MESSAGE, /NONAME, $
          'Python: Array subscript must have same size as source expression.'
      endif
      foreach i, i0, index do begin
        val1 = (nval gt 1) ? value[index] : value
        void = self.__setitem__(i, val1)
      endforeach
    endif else begin
      ; For a scalar index, just set the value.
      void = self.__setitem__(i0, value)
    endelse

    return
  endif

  ; Duck typing to see if we behave like a dictionary
  if (PYTHON_HASATTR(self.pyID, 'keys')) then begin

    ; a[*] or a[0:*] or a[0:-1]
    if (isRange[0]) then begin
      if (i0[0] ne 0 || i0[1] ne -1 || i0[2] ne 1) then $
        MESSAGE, /NONAME, 'Python: Subscript range is not allowed for dict.'
      ; Retrieve all key values pairs
      i0 = self.keys()
    endif

    ; For an array of keys, set all of the keys to the value.
    if (ISA(i0, /ARRAY) || ISA(i0, 'IDL_OBJECT')) then begin
      nval = N_ELEMENTS(value)
      if (nval gt 1 && nval ne N_ELEMENTS(i0)) then begin
        MESSAGE, /NONAME, $
          'Python: Key and Value must have the same number of elements.'
      endif
      index = 0LL
      foreach key, i0 do begin
        val1 = (nval gt 1) ? value[index] : value
        void = self.__setitem__(key, val1)
        index++
      endforeach
    endif else begin
      ; For a scalar key, just set the value.
      void = self.__setitem__(i0, value)
    endelse

    return
  endif

  ; Duck typing - just see if we can index into the object.
  if (~Python_hasattr(self.pyID, "__setitem__")) then begin
    MESSAGE, /NONAME, $
    'Python: Unable to index object of type: "' + pythonClass + '"'
  endif

  shape = PYTHON_GETATTR(self.pyID, "shape")

  ; handle [i:j:k]
  ; These will all fall through to all the lower dimensions.
  switch (n_elements(isRange)) of
    8: i7 = Python._convertSubscript(isRange[7], i7, shape[7])
    7: i6 = Python._convertSubscript(isRange[6], i6, shape[6])
    6: i5 = Python._convertSubscript(isRange[5], i5, shape[5])
    5: i4 = Python._convertSubscript(isRange[4], i4, shape[4])
    4: i3 = Python._convertSubscript(isRange[3], i3, shape[3])
    3: i2 = Python._convertSubscript(isRange[2], i2, shape[2])
    2: i1 = Python._convertSubscript(isRange[1], i1, shape[1])
    1: i0 = Python._convertSubscript(isRange[0], i0, shape[0])
  endswitch

  case n_elements(isRange) of
    1: indices = i0
    2: indices = list(i0, i1)
    3: indices = list(i0, i1, i2)
    4: indices = list(i0, i1, i2, i3)
    5: indices = list(i0, i1, i2, i3, i4)
    6: indices = list(i0, i1, i2, i3, i4, i5)
    7: indices = list(i0, i1, i2, i3, i4, i5, i6)
    8: indices = list(i0, i1, i2, i3, i4, i5, i6, i7)
  endcase

  if (n_elements(isRange) gt 1) then begin
    indices = Python.tuple(indices, /wrap)
  endif

  ; nval = N_ELEMENTS(value)
  ; if (nval gt 1 && nval ne N_ELEMENTS(i0)) then begin
  ;   MESSAGE, /NONAME, $
  ;     'Python: Array subscript must have same size as source expression.'
  ; endif

  ; For ndarray's we can just pass in arrays of indices and values
  ; and Python will take care of filling in the correct values.
  ; If there are multiple subscripts and a single value, the value will
  ; be replicated, which matches IDL's behavior.
  void = self.__setitem__(indices, value)

end


;---------------------------------------------------------------------------
function Python::_overloadBracketsRightSide, isRange, $
  i0, i1, i2, i3, i4, i5, i6, i7

  compile_opt idl2, hidden
  ON_ERROR, 2

  ; If any subscripts are !NULL, return !NULL.
  if (ISA(i0, /NULL) || ISA(i1, /NULL) || ISA(i2, /NULL) || $
    ISA(i3, /NULL) || ISA(i4, /NULL) || ISA(i5, /NULL) || $
    ISA(i6, /NULL) || ISA(i7, /NULL)) then return, !null

  self.GetProperty, CLASSNAME=pythonClass

  if (pythonClass eq 'list' || pythonClass eq 'tuple') then begin

    count = (self._overloadSize())[0]

    ; handle [i:j:k]
    if (isRange[0]) then begin
      ; Adjust negative indices (also handles "*" indices.
      if (i0[0] lt 0 && (ABS(i0[0]) le count)) then i0[0] += count
      if (i0[1] lt 0 && (ABS(i0[1]) le count)) then i0[1] += count
      nelem = (i0[1] - i0[0])/i0[2] + 1
      if (nelem le 0) then begin
        MESSAGE, /NONAME, 'Python: Illegal subscript range.'
      endif
      i0 = LINDGEN(nelem)*i0[2] + i0[0]
    endif

    ; For an array of indices, return an IDL LIST.
    if (ISA(i0, /ARRAY) || ISA(i0, 'IDL_OBJECT')) then begin
      result = LIST()
      foreach i, i0 do begin
        value = self.__getitem__(i)
        result.Add, value, /NO_COPY
      endforeach
    endif else begin
      ; For a scalar index, just return the value.
      result = self.__getitem__(i0)
    endelse

    return, result
  endif

  ; Duck typing to see if we behave like a dictionary
  if (PYTHON_HASATTR(self.pyID, 'keys')) then begin

    ; a[*] or a[0:*] or a[0:-1]
    if (isRange[0]) then begin
      if (i0[0] ne 0 || i0[1] ne -1 || i0[2] ne 1) then $
        MESSAGE, /NONAME, 'Python: Subscript range is not allowed for dict.'
      ; Retrieve all key values pairs
      i0 = self.keys()
    endif

    ; For an array of keys, return an IDL HASH.
    if (ISA(i0, /ARRAY) || ISA(i0, 'IDL_OBJECT')) then begin
      result = HASH()
      foreach key, i0 do begin
        value = self.__getitem__(key)
        result[key] = value
      endforeach
    endif else begin
      ; For a scalar key, just return the value.
      result = self.__getitem__(i0)
    endelse

    return, result
  endif

  ; Duck typing - just see if we can index into the object.
  if (~Python_hasattr(self.pyID, "__getitem__")) then begin
    MESSAGE, /NONAME, `Python: Unable to index object of type: "${pythonClass}"`
  endif

  shape = PYTHON_GETATTR(self.pyID, "shape")

  ; handle [i:j:k]
  ; These will all fall through to all the lower dimensions.
  switch (n_elements(isRange)) of
    8: i7 = Python._convertSubscript(isRange[7], i7, shape[7])
    7: i6 = Python._convertSubscript(isRange[6], i6, shape[6])
    6: i5 = Python._convertSubscript(isRange[5], i5, shape[5])
    5: i4 = Python._convertSubscript(isRange[4], i4, shape[4])
    4: i3 = Python._convertSubscript(isRange[3], i3, shape[3])
    3: i2 = Python._convertSubscript(isRange[2], i2, shape[2])
    2: i1 = Python._convertSubscript(isRange[1], i1, shape[1])
    1: i0 = Python._convertSubscript(isRange[0], i0, shape[0])
  endswitch

  case n_elements(isRange) of
    1: indices = i0
    2: indices = list(i0, i1)
    3: indices = list(i0, i1, i2)
    4: indices = list(i0, i1, i2, i3)
    5: indices = list(i0, i1, i2, i3, i4)
    6: indices = list(i0, i1, i2, i3, i4, i5)
    7: indices = list(i0, i1, i2, i3, i4, i5, i6)
    8: indices = list(i0, i1, i2, i3, i4, i5, i6, i7)
  endcase

  if (n_elements(isRange) gt 1) then begin
    indices = Python.tuple(indices, /wrap)
  endif

  result = self.__getitem__(indices)
  return, result

end


;---------------------------------------------------------------------------
function Python::_overloadForeach, value, index

  compile_opt idl2, hidden
  ON_ERROR, 2

  hasIndex = ISA(index)

  ; If we don't have an index yet, this is our first time through.
  if (~hasIndex) then begin
    self.index = 0
    ; This should throw a useful error if we aren't iterable.
    self.pyIter = Python.iter(self)
    catch, ierr
    if (ierr eq 0) then begin
      obj = self
      ; Now test to see if we have keys, like a Python dict
      self.pyKeys = Python.list(obj.keys())
    endif else begin
      message, /reset
      ; If we don't have keys, then we'll just iterate by number.
      self.pyKeys = Obj_new()
    endelse
    catch, /cancel
  endif

  ; Rather than trying to figure out if we have any elements left,
  ; just catch errors from the "next" call and assume we're done.
  CATCH, ierr
  if (ierr ne 0) then begin
    CATCH, /CANCEL
    MESSAGE, /RESET
    self.pyKeys = Obj_new()
    index = -1
    return, 0b
  endif

  if (Obj_Valid(self.pyKeys)) then begin
    ; If we have keys, then pull out the actual key/value pair.
    ; Set the "index" to the key. It will be returned to the user
    ; but we'll use our stashed self.index in the next foreach call.
    index = self.pyKeys[self.index]
    value = self.__getitem__(index)
  endif else begin
    ; Otherwise the key is just our stashed number
    index = self.index
    ; You might be tempted to use self.__getitem__(index). However, that fails
    ; for things like the numpy.nditer, which doesn't support indexing.
    value = self.pyIter.__next__()
  endelse

  self.index++
  return, 1b
end


;---------------------------------------------------------------------------
function Python::_overloadHelp, varname

  compile_opt idl2, hidden
  ON_ERROR, 2

  myname = STRING(varname, FORMAT='(A-' + STRTRIM(STRLEN(varname) > 15,2) + ',TR1)')

  result = myname + OBJ_CLASS(self) + '  <ID=' + $
    STRTRIM(OBJ_VALID(self,/GET_HEAP_ID),2) + $
    '>'

  if (PYTHON_HASATTR(self.pyID, '__class__')) then begin
    class = PYTHON_GETATTR(self.pyID, '__class__')
    class = PYTHON_CALLMETHOD(-1, 'str', !false, 1, class)
    if (ISA(class, /STRING)) then begin
      result += '  ' + class
    endif
  endif

  if (PYTHON_HASATTR(self.pyID, 'shape')) then begin
    shape = PYTHON_GETATTR(self.pyID, 'shape', /WRAP)
    result += `  shape=${shape}`
  endif

  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadImpliedPrint, varname

  compile_opt idl2, hidden
  ON_ERROR, 2

  ; The "-1" indicates the builtins module
  result = PYTHON_CALLMETHOD(-1, 'repr', !false, 1, self)
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadPrint

  compile_opt idl2, hidden
  ON_ERROR, 2

  ; The "-1" indicates the builtins module
  result = PYTHON_CALLMETHOD(-1, 'str', !false, 1, self)
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadSize

  compile_opt idl2, hidden
  ON_ERROR, 2
  CATCH, iErr
  if (iErr ne 0) then begin
    CATCH, /CANCEL
    MESSAGE, /RESET
    return, 1
  endif

  ; Duck typing - just see if we have a shape attribute (for example, numpy).
  if (Python_hasattr(self.pyID, "shape")) then begin
    shape = PYTHON_GETATTR(self.pyID, "shape")
    if (~isa(shape, 'List')) then begin
      ; This will convert a Pyhton object to a Python list and then an IDL list.
      shape = Python.List(shape)
    endif
    result = (n_elements(shape) gt 0) ? REVERSE(shape.ToArray()) : 1
  endif else begin
    ; The "-1" indicates the builtins module
    result = PYTHON_CALLMETHOD(-1, 'len', !false, 1, self)
    if (result gt 1) then result = [result]
  endelse
  return, result
end


;---------------------------------------------------------------------------
function Python::_callBinaryOperator, arg1, arg2, operator

  compile_opt idl2, hidden
  ON_ERROR, 2

  arg1a = ISA(arg1, 'PYTHON') ? arg1 : Python.Wrap(arg1)
  result = PYTHON_CALLMETHOD(arg1a.pyID, operator, !false, 1, arg2)
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadAsterisk, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__mul__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadCaret, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__pow__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadMinus, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__sub__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadMinusUnary

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self.__neg__()
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadMod, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__mod__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadPlus, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__add__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadSlash, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__truediv__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadIsTrue

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = Python.bool(self)
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadAND, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__and__')
  return, result
end


;----------------------------------------------------------------------------
function Python::_overloadEQ, arg1, arg2
  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__eq__')
  return, result
end


;----------------------------------------------------------------------------
function Python::_overloadGE, arg1, arg2
  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__ge__')
  return, result
end


;----------------------------------------------------------------------------
function Python::_overloadGT, arg1, arg2
  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__gt__')
  return, result
end


;----------------------------------------------------------------------------
function Python::_overloadLE, arg1, arg2
  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__le__')
  return, result
end


;----------------------------------------------------------------------------
function Python::_overloadLT, arg1, arg2
  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__lt__')
  return, result
end


;----------------------------------------------------------------------------
function Python::_overloadNE, arg1, arg2
  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__ne__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadNOT

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self.__invert__()
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadOR, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__or__')
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadTilde

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = ~self->_overloadIsTrue()
  return, result
end


;---------------------------------------------------------------------------
function Python::_overloadXOR, arg1, arg2

  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._callBinaryOperator(arg1, arg2, '__xor__')
  return, result
end


;---------------------------------------------------------------------------
function Python::ToString, format
  compile_opt idl2, hidden
  ON_ERROR, 2

  result = self._overloadImpliedPrint()
  return, result
end


;---------------------------------------------------------------------------
pro Python__define

  compile_opt idl2, hidden

  void = { Python, $
    inherits IDL_Object, $
    pyID: 0LL, $
    pyIter: Obj_New(), $
    pyKeys: Obj_New(), $
    index: 0LL }

end


