;+
; idl-disable potential-undefined-var, potential-var-use-before-def
;-

;  Copyright (c)  NV5 Geospatial Solutions, Inc. All
;        rights reserved. Unauthorized reproduction is prohibited.

; ----------------------------------------------------------------------------
;  cli_progress
;  class to manage and print a progress bar.


;  Initialize:
;  A .Init the sets up the class for use. 
;  However, because this is a class and not a object we wanted to avoid verbiage  
;  that might be confusing.
pro cli_progress::Initialize, _extra = _extra
  compile_opt idl2, hidden, nosave, static
  common ShareIdlProgressBar, _maximum, _title, $
    _width, _complete_char, _pattern, $
    _complete, _lastLength, _incomplete_char, $
    _bartype, _reverse, _isInitialized, $ 
    _autoFinish, _text, _percent, _value, _ticker, $ 
    _remaining, _clock, _speed, _lastUpdate, _previousBar, $
    _remaining_value, _percent_value
    
  on_error, 2
  
  _isInitialized = !true
  
  ; Internal value of out current step
  _value = 0; 
  
  ; Marks the conclusion of a series and determines how many segments to fill in
  _maximum = 100d

  ;  Width of the bar in characters
  _width = 40

  ; Message to be displayed before loading bar
  _title = !null

  ; Message to be displayed after
  _text = !null
  
  ; Keyword flag
  _percent = !true
  
  ; Keyword flag
  _remaining = 0
  
  ; This signifies what type of progress bar we are using.
  _barType = 0
  
  ; Character used for "loaded" chunks
  _complete_char = '#'

  ; Character used for "unloaded" chunks
  _incomplete_char = '-'
  
  ; Pattern to be played before loading bar
  _pattern = [""]
  
  ; Used to track /SPINNER
  _ticker = 0;

  ; Variables related to "ping pong" bars

  ;  reverse keyword means that the load_charcter is supposed to be displayed in order
  ;  for example for the character ,o0 if the bar is headed; 
  ;  ----------->
  ; [....,o0..]
  ; <----------
  ; [...0o,...]
  _reverse = !false

  ; End Variables related to "ping pong" bars
  
  ; Decides if we will autofinish ProgressBars
  _autoFinish = !true
  
  ; Internal flag to say if the progress bar is finished drawing
  _complete = !false

  ; Internal count to handle variable length progress bars
  _lastLength = 0
  
  ; Internal record of the previous loading bar to compare current bar to.
  _previousBar = "" 
  
  ; Internal record of the remaining_value.
  _remaining_value = ""

  ; Set anything the user wants to set
  cli_progress.SetProperty, _extra = _extra

end


;------------------------------------------------------------------------------
; SetProperty:
; used to set preferences and setup defaults.

;  Any changes to progressbar should idealy be done with dot notation, e.g.:
;  cli_progress.maximum=200 or by using the initialize method.
pro cli_progress::SetProperty, MAXIMUM = maximum, TITLE = title, WIDTH = width, $
  LAST_STEP = laststep, COMPLETE_CHAR = complete_char, SPINNER = pattern, $
  TEXT = textint, PERCENT = percent, PRESET = preset, $
  INCOMPLETE_CHAR = incomplete_char, PING_PONG = PING_PONG, REVERSE = reverse, $
  AUTO_FINISH = auto_finish, REMAINING = remaining
  compile_opt idl2, hidden, nosave, static
  common ShareIdlProgressBar
  on_error, 2
  
  if isa(PING_PONG) then begin
    if fix(PING_PONG) eq 1 then begin
      _barType = 1
      _percent = !false
      _maximum = !VALUES.F_INFINITY
      _complete = !False
    endif else begin
      _barType = 0
      _percent = !True
      _maximum = 100
      _complete = !False
    endelse
  endif
  
  if isa(preset) then begin
    if ~isa(preset,/string) then message,"PRESET must be a string."

    presetName = strlowcase(strtrim(preset))
    switch presetName of
      "diamonds": begin
        cli_progress.complete_char   = "◆"
        cli_progress.incomplete_char = "◇"
        break
      end
      "retro": begin
        cli_progress.complete_char   = "="
        cli_progress.incomplete_char = "-"
        cli_progress.spinner = 1
        break
      end
      "ping_pong": begin
        cli_progress.Ping_Pong       = !true
        cli_progress.complete_char   = "<>"
        cli_progress.incomplete_char = "-"
        break
      end
      "shades": begin
        cli_progress.complete_char   = '█'
        cli_progress.incomplete_char = "░"
        break
      end
      
      "squares": begin
        cli_progress.complete_char   = '■'
        cli_progress.incomplete_char = " "
        break
      end
      else: begin
        ; Do nothing
      end
    endswitch
  endif
  
  if isa(maximum) then begin
    _maximum = floor(maximum)
    if _maximum lt 0 then message,"MAXIMUM must be a positive number."
  endif
  
  if isa(title) then begin
    _title = `${title[0]}`
  endif
  
  if isa(pattern) then begin
      
      if isa(pattern, /STRING) then begin
        _pattern = pattern
      endif else begin
        
        if isa(pattern, /NUMBER) then begin
          switch pattern of
            0: begin
              _pattern = [""]
              break 
              end
            1: begin
              _pattern = ["|","/","-","\"]
              break 
              end
            2: begin
              _pattern = ["◴","◷","◶","◵"]
              break 
              end
            3: begin
              _pattern = ["▁","▃","▄","▅","▆","▇","█","▇","▆","▅","▄","▃"]
              break 
              end
            4: begin
              _pattern = ["⠋", "⠙", "⠹", "⠸", "⠼", "⠴", "⠦", "⠧", "⠇", "⠏"]
              break
            end
            else: begin
              ; Do nothing
            end
          endswitch
        endif else begin
          message, "SPINNER must be an array of strings or a preset."
        endelse
      endelse
  endif
  
  if isa(width) then begin
    _width = fix(width)
    if _width lt 0 then message,"WIDTH must be a positive integer."
    _width = width
  endif
  
  if isa(complete_char) then begin
    if ~isa(complete_char, /string) then message, "COMPLETE_CHAR must be a string."
    _complete_char = `${complete_char[0]}`
  endif
  
  if isa(incomplete_char) then begin
    if ~isa(incomplete_char, /string) then  message, "INCOMPLETE_CHAR must be a string."
    _incomplete_char = `${incomplete_char[0]}`
  endif
  
  if isa(reverse) then begin
    _reverse = fix(reverse, TYPE = 2)
  endif
  
  if isa(auto_finish) then begin
    _autoFinish = fix(auto_finish, TYPE = 2)
  endif
  
  if isa(textint) then begin
    _text = `${textint[0]}`
  endif
  
  if isa(percent) then begin
    _percent = keyword_set(percent)
  endif
  
  if isa(remaining) then begin
    _remaining = keyword_set(remaining)
    _speed = 0d
    _clock = 0d
    _lastUpdate = 0d
  endif
end


;------------------------------------------------------------------------------
;START Helper functions for the making of loading bars.
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; This function calculates and formats the percent.
; It returns a string.
function cli_progress::_CalcPercent, step
  compile_opt idl2, hidden, static
  common ShareIdlProgressBar
  on_error, 2
  
  percent = step / _maximum * 100d
  case !true of
    (_maximum gt 10000): StringToAdd = `${percent, "%7.3f"}% `
    (_maximum gt 1000):  StringToAdd = `${percent, "%6.2f"}% `
    (_maximum gt 100):   StringToAdd = `${percent, "%5.1f"}% `
    else: StringToAdd = `${percent, "%3d"}% `
  endcase
  
  _percent_value = StringToAdd
  return, StringToAdd
end


;------------------------------------------------------------------------------
; Take care of cleaning larger than current loading bars.

; This function returns a string. That is the number of spaces required to 
; clean any leftover characters from the old bar.

; Instead of "cleaning" the terminal line at each draw we choose to
; simply re-use it. This is fine until we accidently see data left behind from a
; previous write. This overwrites any of that data with spaces.
; (think partially flushing a pixel buffer)
function cli_progress::_GenerateFiller, currentLength, modifiedPattern
  compile_opt idl2, hidden, static
  common ShareIdlProgressBar
  on_error, 2
  
  modification = ""
  ; Take into account multibyte loading patterns.
  if isa(modifiedPattern) then begin
    if strlen(modifiedPattern) gt 1 then currentLength -= strlen(modifiedPattern) + 1
  endif
  
  if currentLength lt _lastLength then begin
    filler = (' ').dup(strlen(_complete_char))
    modification = STRING((filler).dup(_lastLength - currentLength))
  endif
  
  return, modification
end


;------------------------------------------------------------------------------
; This function calculates the remaining keyword.
; 
; This function returns a string in the format of 
; "<time> `remaining`" to be added to the loading bar.
function cli_progress::_DoRemaining, step
  compile_opt idl2, hidden, static
  common ShareIdlProgressBar
  on_error, 2
  ; If the bar is going to be completed. Don't print remaining. 
  ; This is to give the bar a feeling of completeness.
  if step eq _maximum then return, ""
  
  numSkippedUpdates = double(floor(_MAXIMUM * .05))
  
  if _clock eq 0d then begin
    _clock = SYSTIME(1)
    _lastUpdate = numSkippedUpdates
    return, ""
  endif
  
  if step ge _lastUpdate then begin
    _lastUpdate += numSkippedUpdates
   
    time_since_last_update = SYSTIME(1) - _clock
    _clock = SYSTIME(1)
    
    ; Change weight based on num loops and size of loop. This is a non-standard 
    ; equation to stabilize large progress bars. While also speeding up the 
    ; timer as we approach the end. It was originally weighted to .1.
    alpha = 100d / ( 1000d + (2 * sqrt((_maximum - step) + 1)) )
    
    ; calculate our speed. we use alpha to weight our current speed against a 
    ; "cumulative speed"
    if _speed eq 0 then begin
       _speed = (time_since_last_update * (1-alpha))
    endif else begin
      _speed = (_speed * (1 - alpha)) + (time_since_last_update * alpha)
    endelse
    
  endif
  
  ; While waiting for a big enough sample to measure. return early.
  if _lastUpdate eq numSkippedUpdates then return,""
  
  timeremaining = _speed * (_maximum - step) / numSkippedUpdates
  
  ; Get times in human readable form.
  hour = floor(timeremaining / 3600d)
  minute = floor((timeremaining mod 3600d) / 60)
  second = floor(timeremaining mod 60d)
  
  ; Format said times.
  case (!TRUE) of
    (hour ne 0): estimate = `${hour}h:${minute}m`
    (minute ne 0): estimate = `${minute}m:${second}s`
    else: estimate = `${second lt 1 ? '<' : ''}${second > 1}s`
  endcase
  
  _remaining_value = estimate
  
  return, ` ${estimate} Remaining`
end


;------------------------------------------------------------------------------
;END Helper functions for the making of loading bars.
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;START other helper functions
;------------------------------------------------------------------------------

function prepJSONPackage, barToPass, ERASE = erase, NEWLINE = newline
  compile_opt idl2, hidden, static
  common ShareIdlProgressBar
  on_error, 2
  
  erase_bool = keyword_set(erase)
  newline_bool = keyword_set(newline)

  percent = _percent ? _percent_value : ""
  remaining = _remaining ? _remaining_value : ""
  
  modifiedPattern = _pattern[(_ticker-1) mod n_elements(_pattern)]

  fullHash = orderedHash( $
    "string",         barToPass,        $ ; string: progressBar String
    "erase",          erase_bool,       $ ; bool: did this come from the erase method?
    "newline",        newline_bool,     $ ; bool: did this come from the newline method?
    "bartype",        _bartype,         $ ; integer: 0 = standard bar, 1 = ping-pong spinner
    "complete_char",  _complete_char,   $ ; string: character used for “filled” segments
    "incomplete_char",_incomplete_char, $ ; string: character used for “empty” segments
    "maximum",        _maximum,         $ ; double: total number of steps to reach completion
    "spinner",        modifiedPattern,  $ ; string: single element of the spinners current value. If spinner is not set then "". Will escape "\" as "\\"
    "percent",        percent,          $ ; String: value of percentage complete. If percent is not set then "".
    "remaining",      remaining,        $ ; String: value of time remaining estimate. If percent is not set then "".
    "text",           isa(_text) ? _text : "",   $ ; string: trailing message after the bar. If percent is not set then "".
    "title",          isa(_title) ? _title : "", $ ; string: leading label before the bar. If percent is not set then "".
    "value",          _value,           $ ; integer: current progress step
    "width",          _width )            ; integer: total bar width in characters

  return, JSON_SERIALIZE(fullHash)
end

;------------------------------------------------------------------------------
;END other helper functions
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; Checks to make sure a progress bar is initialized and if its not we go
; ahead and initialize it. We should be doing this before we do anything that
; might use an internal variable.
pro cli_progress::_QuickCheck
  compile_opt idl2, hidden, nosave, static
  common ShareIdlProgressBar
  on_error, 2

  if ~isa(_isInitialized) then begin
    cli_progress.initialize
  endif
end

;------------------------------------------------------------------------------
; Handles the creation of the string that will make up a normal progress bar.
function cli_progress::_MakeProgressBarString, Step
  compile_opt idl2, hidden, nosave, static
  common ShareIdlProgressBar
  on_error, 2
  
  rtrnstr = ""
  
  n = n_elements(_pattern)
  nstars = fix(step / double(_maximum) * double(_width))
  
  ; Assemble basic loading bar.
  loadingBar = string((_complete_char).dup(nstars) + (_incomplete_char).dup(_width - nstars))
  
  ; Add pattern if pattern is desired.
  if fix(total(strlen(_pattern))) gt 0 then begin
    modifiedPattern = _pattern[_ticker mod n]
    rtrnstr += `[${modifiedPattern}] `
    _ticker += 1
  endif
  
  ; Add title if title is desired.
  if (isa(_title) && (_title ne '')) then begin
    if strlen(`${_title}`) gt 0 then rtrnstr = rtrnstr+`${_title} `
  endif
  
  ; Add percent provided percent is not set to false.
  if (_percent) then begin
    rtrnstr += cli_progress._CalcPercent(step)
  endif
  
  ; Add Loading bar.
  rtrnstr = `${rtrnstr}[${loadingBar}]`
  
  ; Anything after this will be after the body of the loading bar.
  
  ; Add remaining if remaining is desired.
  if _remaining then begin
    rtrnstr += cli_progress._DoRemaining(step)
  endif
  
  ; Finally add any text the user might want to add.
  if (isa(_text) && (_text ne '')) then begin
    rtrnstr += ` ${_text}`
  endif
  
  ; Do we need filler? if so fill.
  currentLength = strlen(rtrnstr)
  rtrnstr += cli_progress._GenerateFiller(currentLength, modifiedPattern)
  _lastLength = currentLength
  
  return, rtrnstr
end


;------------------------------------------------------------------------------
; Handles the creation of the string that will make up a "ping pong" progress bar.
function cli_progress::_MakePing_PongBarString, Step
  compile_opt idl2, hidden, nosave, static
  common ShareIdlProgressBar
  on_error, 2
  
  ; Prep contents of the loading bar.
  section = fix(Step / _width) + 1
  n = n_elements(_pattern)
  
  ; barStep will be the index where we are in the loading bar.
  ; So if the width is 30 it will count 0-30.
  barStep = Step mod _width
  
  ; We want to move to the right on odd and to the left on even.
  odd = section mod 2
  
  if odd then begin 
    loadingBar = string(_incomplete_char.dup(barStep) + $
                 _complete_char + _incomplete_char.dup(_width - barStep))
  endif else begin
    ; Else (even) move left .
    modLoadCharacter = _complete_char
    
    if _reverse eq 1 then begin
      ; Handle _reverse keyword.
      modLoadCharacter = modLoadCharacter.reverse()
    endif
    
    loadingBar = string(_incomplete_char.dup(_width - barStep) + $
                 modLoadCharacter + _incomplete_char.dup(barStep))     
  endelse
  
  ; Format loading pattern.
  rtrnstr = ""
  
  ; Add pattern if pattern is desired.
  if fix(total(strlen(_pattern))) gt 0 then begin
    modifiedPattern = _pattern[_ticker mod n]
    rtrnstr += `[${modifiedPattern}] `
    _ticker += 1
  endif
  
  ; Add title if title is desired.
  if (isa(_title) && (_title ne '')) then begin
    rtrnstr += `${_title} `
  endif
  
  ; Add percent provided percent is not set to false.
  if (_percent && (_maximum ne !VALUES.F_INFINITY)) then begin
    rtrnstr +=  cli_progress._CalcPercent(step)
  endif
  
  ; Add the Loading bar.
  rtrnstr = `${rtrnstr}[${loadingBar}]`
  
  ; Anything after this point will be after  the loading bar.
  
  ; Add remaining if remaining is desired.
  if (_remaining && (_maximum ne !VALUES.F_INFINITY)) then begin
    rtrnstr += cli_progress._DoRemaining(step)
  endif

  ; Finally add any text the user might want to add.
  if (isa(_text) && (_text ne '')) then begin
    rtrnstr += ` ${_text}`
  endif
  
  ; do we need filler? if so fill.
  currentLength = strlen(rtrnstr)
  rtrnstr += cli_progress._GenerateFiller(currentLength, modifiedPattern)
  _lastLength = currentLength

  return, rtrnstr
end


;------------------------------------------------------------------------------
; Update:
; Updates the progress bar.
; The creation of the progress bar string to the corresponding bar type.
pro cli_progress::Update, step, TEXT = text, OUTPUT = output
  compile_opt idl2, nosave, static
  common ShareIdlProgressBar
  on_error, 2
  
  ; Check to make sure we are initialized.
  cli_progress._quickCheck
  
  if isa(text) then cli_progress.SetProperty, TEXT = text
  _value = isa(step) ? step : _value + 1
  
  stepCopy = _value
  
  if stepCopy le _maximum then begin
  ; don't print the bar if the bar is finished.
    if _autoFinish then begin
      willBeCompleted = (_value eq _maximum)
      
      if (_complete && willBeCompleted) then begin
        return
      endif
      _complete = willBeCompleted
    endif

    switch _barType of
      0: begin
        progressBarString = cli_progress._makeProgressBarString(double(stepCopy))
        break
      end
      1: begin
        progressBarString = cli_progress._makePing_PongBarString(double(stepCopy))
        break
      end
      else: message, "non-Valid bar type."
    endswitch
    
    if ~arg_present(output) then begin
      
      if progressBarString ne _previousBar then begin
        ; if IDL_IS_IDL_MACHINE is set then send off the progress bar through the pipe.
        ; This is so the vsCode extension can handle the progress bar itself.
        if getenv('IDL_IS_IDL_MACHINE') ne '' then begin
          jsonForVSCODE = prepJSONPackage(progressBarString)
          void = IDLNotify('cli_progressNotification', jsonForVSCODE)
          _lastLength = 0
        endif else begin
          print, `\r${progressBarString}`, newline = _complete
        endelse
        _previousBar = progressBarString
      endif
              
    endif else begin
      output = progressBarString
      _lastLength = 0
      _previousBar = ""
    endelse
    
  endif
  ; if we have completed then we have carrage returned.(/n) 
  ; then we dont have any "buffer" to clear. Please see _GenerateFiller
  if _complete then _lastLength = 0
end


;---------------------------------------------------------------------------------
; Newline:
; Its use is to "break" the progress bar. Returns the Console to a state the
; user expects.
pro cli_progress::Newline
  compile_opt idl2, nosave, static
  on_error, 2
  if getenv('IDL_IS_IDL_MACHINE') ne '' then begin
    jsonForVSCODE = prepJSONPackage("", /NEWLINE)
    void = IDLNotify('cli_progressNotification',"")
  endif else begin
    print, ""
  endelse
  
   _previousBar = ""
end


;------------------------------------------------------------------------------
; Erase:
; Used to erase the progress bar and return the terminal to the user with the
; cursor at the begining of the same line.
pro cli_progress::Erase
  compile_opt idl2, nosave, static
  common ShareIdlProgressBar
  on_error, 2
  
  if fix(total(strlen(_pattern))) gt 0 then begin
    n = n_elements(_pattern)
    modifiedPattern = _pattern[_ticker mod n]
  endif
  
  buffer = _lastLength
  
  if isa(modifiedPattern) then begin
    buffer += strlen(modifiedPattern) + 1
  endif
  
  erasedBar = `\r${(" ").dup(buffer)}\r`
  
  if getenv('IDL_IS_IDL_MACHINE') ne '' then begin
    jsonForVSCODE = prepJSONPackage(erasedBar, /ERASE)
    void = IDLNotify('cli_progressNotification', erasedBar)
  endif else begin
    print, erasedBar, newline = 0
  endelse
  
  _previousBar = ""
end


;------------------------------------------------------------------------------
; Finish:
; Finishes a progress bar no matter the progess.
; If the _autoFinish keyword is set. Then this and Newline will be the only way
; to stop the progress bar.
pro cli_progress::Finish, OUTPUT = output, TEXT = text
  compile_opt idl2, nosave, static
  common ShareIdlProgressBar
  on_error, 2
  
  ; Check to make sure we are initialized.
  cli_progress._quickCheck

  ; If someone is finishing over and over dont print over and over.
  if _complete then return
    
    ; Dont try and hold for a autoFinish as we are finishing now.
    autoFinishStore = _autoFinish
    _autoFinish = !false
    _complete = !true
  
    switch _barType of
      0: begin
        if ~arg_present(output) then begin
          cli_progress.update, _maximum, TEXT = text
        endif else begin
          cli_progress.update, _maximum, OUTPUT = output, TEXT = text
        endelse
        break
      end
      1: begin
        step = min([_value,_maximum])
        if ~arg_present(output) then begin
          cli_progress.update, step, TEXT = text
        endif else begin
          cli_progress.update, step, OUTPUT = output, TEXT = text
        endelse
        break
      end
      else: message, "non-Valid bar type."
    endswitch
    
    ; Restore autofinish.
    _autoFinish = autoFinishStore
end


;-----------------------------------------------------------------------------
pro cli_progress__define
compile_opt idl2, hidden, nosave
  !null = {cli_progress, inherits IDL_Object}
end
