; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;
;----------------------------------------------------------------------------
; :Description:
;    The YAML_PARSE function takes a YAML filename or string and converts it
;    into an IDL variable.
;
; :Returns: float
;   
; :Params:
;    String:
;      *String* must be a:
;         - Valid YAML string or string array
;         - Path to Filename of a valid YAML file 
;
; :Keywords:
;    PRESERVE_ALIAS:
;       By default, YAML_PARSE automatically resolves all YAML aliases
;       from their corresponding anchors and returns the resulting values,
;       including any value overrides. If PRESERVE_ALIAS is set then the
;       result will contain the original, unmodified aliases as strings.
;
; :Author:
;   CT, March 2023.
;-
function YAML_Parse, yaml, PRESERVE_ALIAS=preserve

  compile_opt idl2, hidden
  ON_ERROR, 2
  
  CATCH, iErr
  if (iErr ne 0) then begin
    CATCH, /CANCEL
    msg = (!ERROR_STATE.MSG).Replace('YAML_PARSE_INTERNAL: ', '')
    message, msg
  endif

  ; See if our input is actually a file name.
  if (ISA(yaml, 'STRING', /SCALAR) && FILE_TEST(yaml, /READ)) then begin
    data = string(read_binary(yaml))
  endif else begin
    data = (string(yaml)).join(10b)
  endelse

  result = yaml_parse_internal(data, PRESERVE_ALIAS=preserve)

  if (isa(result, 'YAML_Multidoc') || isa(result, 'YAML_Stream_Map') || $
    isa(result, 'YAML_Stream_Sequence')) then begin
    str = data.split(`\n`)
    comments = str[where(str.startsWith('#') or str.startsWith('%YAML'), /null)]
    if (isa(comments)) then begin
      result.SetProperty, COMMENTS=comments
    endif
  endif
  return, result
end

