; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;+
; Name:
;   iRESOLVE
;
; Purpose:
;   Resolves all IDL code within the iTools directory, as well
;   as all other necessary IDL code. Useful for constructing save
;   files containing user code that requires the iTools framework.
;
; Arguments:
;   None.
;
; Keywords:
;   PATH: Set this keyword to a string giving the full path to the iTools
;       directory. The default is to use the lib/itools subdirectory
;       within which the iRESOLVE procedure resides.
;
; MODIFICATION HISTORY:
;   Written by:  CT, RSI, June 2003
;   Modified:  CT, VIS, April 2014: Ignore files with __ in their name.
;


;-------------------------------------------------------------------------
pro iresolve, PATH=pathIn

    compile_opt idl2, hidden

    if (N_ELEMENTS(pathIn) gt 0) then begin
        path = pathIn
    endif else begin
        ; Assume this program is in a subdirectory of iTools.
        path = FILE_DIRNAME((ROUTINE_INFO('iresolve', $
            /SOURCE)).path, /MARK_DIR)
        ; Add Graphics directory
        path =[path, FILE_SEARCH(path[0]+'../graphics', /MARK_DIRECTORY)]
        ; Add Datatypes directory
        path =[path, FILE_SEARCH(path[0]+'../datatypes', /MARK_DIRECTORY)]
    endelse

    filenames = FILE_SEARCH(path, '*.pro', /FULLY_QUALIFY)

    ; Files which we don't need (or can't) compile.
    excludelist=[ $
      'cw_iterror', $   ;  @ includes
      'graphic_error', $
      'idlit_catch', $
      'idlit_on_error2', $
      'idlit_itoolerror', $
      'idlitconfig', $
      '_idlitcreatesave', $  ; don't include ourself
      ; ENVI files
      'export_envi__define', $
      'export_envioview__define', $
      'idlcfgrwinscene__define', $
      'idlcfembeddeduienvi', $
      'idlcfuienvieclipseadaptor__define', $
      ; Resolve routines
      'resolve_all', $
      'resolve_all_body', $
      'resolve_all_class', $
      ; Datatypes
      'mksav_datatypes']

    ; These are classes which are outside of the iTools directory.
    classlist = [ $
        'idlfflangcat', $
        'idlgrtextedit', $
        'trackball']

    filenames = FILE_BASENAME(filenames, '.pro')

    for i=0,N_ELEMENTS(excludelist)-1 do $
        filenames = filenames[WHERE(filenames ne excludelist[i])]

    ; We can't compile methods by themselves. So remove them from the list.
    ; Then, for classes with outside methods, add them to the special classlist.
    spos = filenames.IndexOf('__')
    methodsOutsideClasses = $
      WHERE(spos gt 0 and ~filenames.EndsWith('__define'), COMPLEMENT=keep)
    spos = spos[methodsOutsideClasses]
    classesWithOutsideMethods = $
      ((filenames[methodsOutsideClasses]).Substring(0,spos-1)).Uniq()
    classlist = ([classlist, classesWithOutsideMethods]).Uniq()
    filenames = filenames[keep]

    RESOLVE_ROUTINE, filenames, /EITHER, $
        /COMPILE_FULL_FILE, /NO_RECOMPILE

    ; Determine if we are being called from within RESOLVE_ALL.
    ; In this case, just resolve our special classes and assume
    ; that RESOLVE_ALL will pick up everything else.
    scope = SCOPE_TRACEBACK()
    withinResolveAll = MAX(STRPOS(scope, 'RESOLVE_ALL')) ge 0

    if (withinResolveAll) then begin
      RESOLVE_ALL_CLASS, classlist, 1, 1, !null
    endif else begin
      ; Call RESOLVE_ALL to resolve our classes plus everything else.
      RESOLVE_ALL, CLASS=classlist, /QUIET
    endelse
    
    ; Add read and query routines from iGetReaders
    ; These routines are listed in the idlextensions.xml file and might not get
    ; compiled through any use of RESOLVE_*
    RESOLVE_ROUTINE, 'igetreaders', /EITHER, $
      /COMPILE_FULL_FILE, /NO_RECOMPILE
    readers = iGetReaders()
    for i=0,N_ELEMENTS(readers[0,*])-1 do begin
      catch, err
      if (err eq 0) then begin
        ; Resolve reader routine
        RESOLVE_ROUTINE, readers[0,i], /EITHER, $
            /COMPILE_FULL_FILE, /NO_RECOMPILE

        ; Resolve any additional routines that are accessed 
        ; via CALL_[PROCEDURE|FUNCTION].
        ; This list has to be hardcoded for now
        if (STRPOS(STRUPCASE(readers[0,i]), 'HDF5') ne -1) then $
          RESOLVE_ROUTINE, 'h5_parse', /EITHER, $
            /COMPILE_FULL_FILE, /NO_RECOMPILE
        if (STRPOS(STRUPCASE(readers[0,i]), 'READ_IMAGE') ne -1) then $
          RESOLVE_ROUTINE, 'query_image', /EITHER, $
            /COMPILE_FULL_FILE, /NO_RECOMPILE

        ; Resolve query routine
        if (readers[4,i] ne '') then $
          RESOLVE_ROUTINE, readers[4,i], /EITHER, $
              /COMPILE_FULL_FILE, /NO_RECOMPILE
      endif else begin
        ; If read/query routine is not in the current path, or is C code,
        ; handle error and continue
        catch, /cancel
        message, /reset
        continue
      endelse
    endfor
    
    ; Restore any save files
    filenames = FILE_SEARCH(path, '*.sav', /FULLY_QUALIFY)
    if (MAX(filenames ne '') ne 0) then begin
      for i=0,N_ELEMENTS(filenames)-1 do begin
        RESTORE, filenames[i]
      endfor
    endif
    
end

